package hostinfo

import (
	"encoding/json"
	"hash/fnv"
	"io"
	"io/ioutil"
	"os"
	"strings"

	"github.com/golang/protobuf/ptypes"

	pb "a.yandex-team.ru/infra/hostctl/proto"
)

const (
	ServerInfoFile = "/etc/server_info.json"
	BootIDFile     = "/proc/sys/kernel/random/boot_id"
)

type ServerInfo struct {
	GencfgGroups  []string `json:"gencfg"`
	KernelRelease string   `json:"kernelrelease"`
	CPUModel      string   `json:"cpu_model"`
	Location      string   `json:"location"`
	NodeName      string   `json:"nodename"`
	DC            string   `json:"walle_dc"`
	WalleProject  string   `json:"walle_project"`
	WalleSwitch   string   `json:"walle_switch"`
	WalleTags     []string `json:"walle_tags"`
	MemTotal      int32    `json:"mem_total"`
	WalleQueue    string   `json:"walle_queue"`
	OSCodename    string   `json:"oscodename"`
	OSArch        string   `json:"osarch"`
}

func readServerInfo(file io.Reader) (*ServerInfo, error) {
	content, err := ioutil.ReadAll(file)
	if err != nil {
		return nil, err
	}
	info := &ServerInfo{}
	if err = json.Unmarshal(content, info); err != nil {
		return nil, err
	}
	return info, nil
}

// Generates consistent number [0, 99] for provided server
// we using Fowler–Noll–Vo_hash_function algorithm to generate int32 number
// than we get mod 100 and add 100 if value was negative
// https://en.wikipedia.org/wiki/Fowler%E2%80%93Noll%E2%80%93Vo_hash_function
func genServerNum(hostname string) (int32, error) {
	h := fnv.New32a()
	_, _ = h.Write([]byte(hostname))
	n := int32(h.Sum32())
	n %= 100
	if n < 0 {
		n += 100
	}
	return n, nil
}

type StatReader interface {
	io.Reader
	Stat() (os.FileInfo, error)
}

// FromStatReader reads HostInfo from StatReader (usually - os.File)
func FromStatReader(reader StatReader) (*pb.HostInfo, error) {
	info, err := readServerInfo(reader)
	if err != nil {
		return nil, err
	}
	stat, err := reader.Stat()
	if err != nil {
		return nil, err
	}
	mtime, err := ptypes.TimestampProto(stat.ModTime())
	if err != nil {
		return nil, err
	}
	num, err := genServerNum(info.NodeName)
	if err != nil {
		return nil, err
	}
	return &pb.HostInfo{
		Hostname:      info.NodeName,
		Num:           num,
		WalleProject:  info.WalleProject,
		WalleTags:     info.WalleTags,
		NetSwitch:     info.WalleSwitch,
		GencfgGroups:  info.GencfgGroups,
		Location:      info.Location,
		Dc:            info.DC,
		KernelRelease: info.KernelRelease,
		CpuModel:      info.CPUModel,
		Mtime:         mtime,
		MemTotalMib:   info.MemTotal,
		DcQueue:       info.WalleQueue,
		OsCodename:    info.OSCodename,
		OsArch:        info.OSArch,
	}, nil
}

func BootID() (string, error) {
	f, err := os.Open(BootIDFile)
	if err != nil {
		return "", err
	}
	bootID, err := io.ReadAll(f)
	return strings.Trim(string(bootID), " \n"), err
}
