package hostinfo

import (
	"fmt"
	"io"
	"os"
	"reflect"
	"strings"
	"testing"
	"time"

	"github.com/golang/protobuf/ptypes"
	"github.com/stretchr/testify/assert"

	pb "a.yandex-team.ru/infra/hostctl/proto"
)

const (
	serverInfoTest = `{
    "cpu_model": "Intel(R) Xeon(R) CPU E5-2650 v2 @ 2.60GHz",
    "cpuarch": "x86_64",
    "disks_info": {
        "nonssd": [
            "/dev/sdc"
        ],
        "nvme": [],
        "ssd": [
            "/dev/sda",
            "/dev/sdb"
        ]
    },
    "gencfg": [
        "ALL_RTC",
        "ALL_RUNTIME",
        "ALL_SEARCH",
        "MAN_JUGGLER_CLIENT_STABLE",
        "MAN_KERNEL_UPDATE_3",
        "MAN_RTC_SLA_TENTACLES_PROD",
        "MAN_RUNTIME",
        "MAN_SAAS_CLOUD",
        "MAN_SAAS_CLOUD_BLACKLISTED_REFRESH",
        "MAN_SAAS_CLOUD_SHMICK",
        "MAN_SEARCH",
        "MAN_YASM_YASMAGENT_STABLE"
    ],
    "id": "man1-3720.search.yandex.net",
    "init": "systemd",
    "kernel": "Linux",
    "kernelrelease": "4.19.138-35",
    "lldp": [
        {
            "port": "40GE1/0/6:2",
            "switch": "man1-s66"
        }
    ],
    "location": "man",
    "lsb_distrib_codename": "xenial",
    "lsb_distrib_description": "Ubuntu 16.04.5 LTS",
    "lsb_distrib_id": "Ubuntu",
    "lsb_distrib_release": "16.04",
    "lui": {
        "name": "web",
        "timestamp": 1586223614
    },
    "mem_total": 257920,
    "nodename": "man1-3720.search.yandex.net",
    "num_cpus": 32,
    "os": "Ubuntu",
    "os_family": "Debian",
    "osarch": "amd64",
    "oscodename": "xenial",
    "osfinger": "Ubuntu-16.04",
    "osfullname": "Ubuntu",
    "osmajorrelease": "16",
    "osrelease": "16.04",
    "osrelease_info": [
        16,
        4
    ],
    "points_info": {
        "/place": {
            "device": "/dev/sdc4",
            "disks_info": [
                {
                    "sdc": {
                        "disk_type": "hdd"
                    }
                }
            ],
            "mount_point": "/place",
            "raid": null
        },
        "/ssd": {
            "device": "/dev/md5",
            "disks_info": [
                {
                    "sdb": {
                        "disk_type": "ssd"
                    }
                },
                {
                    "sda": {
                        "disk_type": "ssd"
                    }
                }
            ],
            "mount_point": "/ssd",
            "raid": "raid0"
        }
    },
    "walle_country": "fi",
    "walle_dc": "man",
    "walle_location": "man",
    "walle_project": "rtc-prestable",
    "walle_queue": "man2",
    "walle_rack": "2D16",
    "walle_switch": "man1-s66",
    "walle_tags": [
        "rtc",
        "rtc.automation-enabled",
        "rtc.gpu-none",
        "rtc.reboot_segment-prestable-def",
        "rtc.scheduler-gencfg",
        "rtc.stage-prestable",
        "rtc_network",
        "runtime",
        "search",
        "skynet_installed",
        "yasm_monitored"
    ]
}
`
)

func Test_readServerInfo(t *testing.T) {
	type args struct {
		file io.Reader
	}
	tests := []struct {
		name    string
		args    args
		want    *ServerInfo
		wantErr bool
	}{{
		name: "valid file",
		args: args{
			file: strings.NewReader(serverInfoTest),
		},
		want: &ServerInfo{
			GencfgGroups: []string{
				"ALL_RTC",
				"ALL_RUNTIME",
				"ALL_SEARCH",
				"MAN_JUGGLER_CLIENT_STABLE",
				"MAN_KERNEL_UPDATE_3",
				"MAN_RTC_SLA_TENTACLES_PROD",
				"MAN_RUNTIME",
				"MAN_SAAS_CLOUD",
				"MAN_SAAS_CLOUD_BLACKLISTED_REFRESH",
				"MAN_SAAS_CLOUD_SHMICK",
				"MAN_SEARCH",
				"MAN_YASM_YASMAGENT_STABLE"},
			KernelRelease: "4.19.138-35",
			Location:      "man",
			NodeName:      "man1-3720.search.yandex.net",
			DC:            "man",
			WalleProject:  "rtc-prestable",
			WalleSwitch:   "man1-s66",
			WalleTags: []string{
				"rtc",
				"rtc.automation-enabled",
				"rtc.gpu-none",
				"rtc.reboot_segment-prestable-def",
				"rtc.scheduler-gencfg",
				"rtc.stage-prestable",
				"rtc_network",
				"runtime",
				"search",
				"skynet_installed",
				"yasm_monitored"},
			CPUModel:   "Intel(R) Xeon(R) CPU E5-2650 v2 @ 2.60GHz",
			MemTotal:   257920,
			WalleQueue: "man2",
			OSCodename: "xenial",
			OSArch:     "amd64",
		},
		wantErr: false,
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := readServerInfo(tt.args.file)
			if (err != nil) != tt.wantErr {
				t.Errorf("readServerInfo() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("readServerInfo() got = %v, want %v", got, tt.want)
			}
		})
	}
}

type serverInfoReaderMock struct {
	buf   io.Reader
	mtime time.Time
}

func (r *serverInfoReaderMock) Name() string {
	panic("implement me")
}

func (r *serverInfoReaderMock) Size() int64 {
	panic("implement me")
}

func (r *serverInfoReaderMock) Mode() os.FileMode {
	panic("implement me")
}

func (r *serverInfoReaderMock) ModTime() time.Time {
	return r.mtime
}

func (r *serverInfoReaderMock) IsDir() bool {
	panic("implement me")
}

func (r *serverInfoReaderMock) Sys() interface{} {
	panic("implement me")
}

func newServerInfoReaderMock(content string, mtime time.Time) *serverInfoReaderMock {
	return &serverInfoReaderMock{
		buf:   strings.NewReader(content),
		mtime: mtime,
	}
}

func (r *serverInfoReaderMock) Read(p []byte) (n int, err error) {
	return r.buf.Read(p)
}

func (r *serverInfoReaderMock) Stat() (os.FileInfo, error) {
	return r, nil
}

func TestFromServerInfo(t *testing.T) {
	info := &ServerInfo{
		GencfgGroups: []string{
			"ALL_RTC",
			"ALL_RUNTIME",
			"ALL_SEARCH",
			"MAN_JUGGLER_CLIENT_STABLE",
			"MAN_KERNEL_UPDATE_3",
			"MAN_RTC_SLA_TENTACLES_PROD",
			"MAN_RUNTIME",
			"MAN_SAAS_CLOUD",
			"MAN_SAAS_CLOUD_BLACKLISTED_REFRESH",
			"MAN_SAAS_CLOUD_SHMICK",
			"MAN_SEARCH",
			"MAN_YASM_YASMAGENT_STABLE"},
		KernelRelease: "4.19.138-35",
		Location:      "man",
		NodeName:      "man1-3720.search.yandex.net",
		DC:            "man",
		WalleProject:  "rtc-prestable",
		WalleSwitch:   "man1-s66",
		WalleTags: []string{
			"rtc",
			"rtc.automation-enabled",
			"rtc.gpu-none",
			"rtc.reboot_segment-prestable-def",
			"rtc.scheduler-gencfg",
			"rtc.stage-prestable",
			"rtc_network",
			"runtime",
			"search",
			"skynet_installed",
			"yasm_monitored"},
		CPUModel:   "Intel(R) Xeon(R) CPU E5-2650 v2 @ 2.60GHz",
		MemTotal:   257920,
		WalleQueue: "man2",
		OSCodename: "xenial",
		OSArch:     "amd64",
	}
	num, _ := genServerNum(info.NodeName)
	mock := newServerInfoReaderMock(serverInfoTest, time.Time{})
	mtime, _ := ptypes.TimestampProto(time.Time{})
	expected := &pb.HostInfo{
		Hostname:      info.NodeName,
		Num:           num,
		WalleProject:  info.WalleProject,
		WalleTags:     info.WalleTags,
		NetSwitch:     info.WalleSwitch,
		GencfgGroups:  info.GencfgGroups,
		Location:      info.Location,
		Dc:            info.DC,
		KernelRelease: info.KernelRelease,
		CpuModel:      info.CPUModel,
		Mtime:         mtime,
		MemTotalMib:   info.MemTotal,
		DcQueue:       info.WalleQueue,
		OsCodename:    info.OSCodename,
		OsArch:        info.OSArch,
	}
	hi, err := FromStatReader(mock)
	assert.NoError(t, err)
	assert.Equal(t, expected, hi)
}

func generateHostname(dc string, num int) string {
	return fmt.Sprintf("%s-%04d.search.yandex.net", dc, num)
}

func sum(nums map[int32]int) int {
	t := 0
	for _, c := range nums {
		t += c
	}
	return t
}

func Test_genServerNum(t *testing.T) {
	nums := make(map[int32]int)
	// generate distribution of nums
	for _, dc := range []string{"sas", "vla", "man", "iva", "myt"} {
		for j := 0; j < 5; j++ {
			for i := 1; i < 10000; i++ {
				fullDc := fmt.Sprintf("%s%d", dc, j)
				hostname := generateHostname(fullDc, i)
				num, err := genServerNum(hostname)
				if err != nil {
					t.Error(err)
				}
				nums[num]++
			}
		}
	}
	total := sum(nums)
	errs := make([]string, 0, 100)
	eps := 0.05
	for num, count := range nums {
		dist := float64(count) / (float64(total) / 100)
		if dist > eps+1 ||
			dist < -(eps+1) {
			errs = append(errs, fmt.Sprintf("diviation of nums['%d']=%.6f more than %.2f", num, dist, eps))
		}
	}
	if len(errs) > 0 {
		t.Errorf("Invalid distribution of genServerNum\n"+
			"filled nums count=%d:\n"+
			"errored nums count=%d:\n"+
			"\t%s", len(nums), len(errs), strings.Join(errs, ";\n\t"))
	}
	for n := range nums {
		if n < 0 || n >= 100 {
			t.Errorf("genServerNum returns value not from [0, 99]: %d", n)
		}
	}
}

// fixing distribution on 3  example hostnames
// if we will change distribution we will notice that from this test
func Test_genServerNumNotChanged(t *testing.T) {
	h := generateHostname("man1", 1)
	n, _ := genServerNum(h)
	num := 66
	if int(n) > num {
		t.Errorf("num for '%s' should be less or eq than %d but was %d", h, num, n)
	}
	h = generateHostname("sas0", 4534)
	n, _ = genServerNum(h)
	num = 37
	if int(n) > num {
		t.Errorf("num for '%s' should be less or eq than %d but was %d", h, num, n)
	}
	h = generateHostname("iva3", 9424)
	n, _ = genServerNum(h)
	num = 94
	if int(n) > num {
		t.Errorf("num for '%s' should be less or eq than %d but was %d", h, num, n)
	}
}
