package infra

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"path"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/httputil/headers"
)

const (
	URL          = "https://infra-api.yandex-team.ru/"
	RtcServiceID = 298
)

type Location string

const (
	SasLocation Location = "SAS"
	VlaLocation Location = "VLA"
	ManLocation Location = "MAN"
	MskLocation Location = "MSK"
	AllLocation Location = "ALL"
)

func (l Location) EnvironmentID() EnvironmentID {
	switch l {
	case SasLocation:
		return sasEnvironmentID
	case VlaLocation:
		return vlaEnvironmentID
	case ManLocation:
		return manEnvironmentID
	case MskLocation:
		return mskEnvironmentID
	case AllLocation:
		return allEnvironmentID
	}
	panic(fmt.Sprintf("invalid location: %s", l))
}

type EnvironmentID int

const (
	sasEnvironmentID EnvironmentID = 390
	vlaEnvironmentID EnvironmentID = 391
	manEnvironmentID EnvironmentID = 388
	mskEnvironmentID EnvironmentID = 389
	allEnvironmentID EnvironmentID = 564
)

type Client struct {
	httpClient *http.Client
	u          *url.URL
	oauthToken string
}

func NewClient(oauthToken string) *Client {
	u, _ := url.Parse(URL)
	return &Client{
		httpClient: &http.Client{
			Transport: http.DefaultTransport,
			Timeout:   30 * time.Second,
		},
		u:          u,
		oauthToken: oauthToken,
	}
}

type DeploymentEvent struct {
	Name        string
	Location    Location
	Description string
	StartTime   time.Time
	Duration    time.Duration
	Percent     int
}

func (r *DeploymentEvent) Event() *Event {
	title := fmt.Sprintf("%s on %d %s", r.Name, r.Percent, r.Location)
	if r.Percent == 0 {
		title = fmt.Sprintf("%s on %s", r.Name, r.Location)
	}
	req := &Event{
		Title:         title,
		Description:   r.Description,
		EnvironmentID: int(r.Location.EnvironmentID()),
		ServiceID:     RtcServiceID,
		StartTime:     int(r.StartTime.Unix()),
		FinishTime:    int(r.StartTime.Add(r.Duration).Unix()),
		Type:          "maintenance",
		Severity:      "minor",
		Meta:          map[string]string{},
	}
	switch r.Location {
	case SasLocation:
		req.Sas = true
	case VlaLocation:
		req.Vla = true
	case ManLocation:
		req.Man = true
	case MskLocation:
		req.Myt = true
		req.Iva = true
	case AllLocation:
		req.Sas = true
		req.Vla = true
		req.Man = true
		req.Myt = true
		req.Iva = true
	}
	return req
}

func (c *Client) CreateDeploymentEvent(r *DeploymentEvent) error {
	return c.createEvent(r.Event())
}

type Event struct {
	Title                  string            `json:"title"`
	Description            string            `json:"description"`
	EnvironmentID          int               `json:"environmentId"`
	ServiceID              int               `json:"serviceId"`
	StartTime              int               `json:"startTime"`
	FinishTime             int               `json:"finishTime"`
	Type                   string            `json:"type"`
	Severity               string            `json:"severity"`
	Man                    bool              `json:"man"`
	Myt                    bool              `json:"myt"`
	Sas                    bool              `json:"sas"`
	Vla                    bool              `json:"vla"`
	Iva                    bool              `json:"iva"`
	Tickets                string            `json:"tickets"`
	Meta                   map[string]string `json:"meta"`
	SendEmailNotifications bool              `json:"sendEmailNotifications"`
	SetAllAvailableDc      bool              `json:"setAllAvailableDc"`
}

func (c *Client) createEvent(r *Event) error {
	jsonReq, err := json.Marshal(r)
	if err != nil {
		return err
	}
	eventURL := *(c.u)
	eventURL.Path = path.Join(eventURL.Path, "v1", "events")
	_, err = c.httpClient.Do(&http.Request{
		Method: "POST",
		URL:    &eventURL,
		Header: http.Header{
			headers.ContentTypeKey:   []string{headers.TypeApplicationJSON.String()},
			headers.AuthorizationKey: []string{strings.Join([]string{"OAuth", c.oauthToken}, " ")},
		},
		Body: io.NopCloser(bytes.NewBuffer(jsonReq)),
	})
	return err
}
