// Package render used for generic hostctl unit rendering.
package render

import (
	"io"
	"path"
	"path/filepath"

	"a.yandex-team.ru/infra/hostctl/internal/template"
	"a.yandex-team.ru/infra/hostctl/internal/unit"
	"a.yandex-team.ru/infra/hostctl/pkg/unitstorage"

	"google.golang.org/protobuf/proto"

	hostpb "a.yandex-team.ru/infra/hostctl/proto"
)

// BUG(warwish): package cannot render fragmented units.
// Broken functions marked as deprecated. Fix will be provided soon.

// Result used to represent result of rendering operation.
type Result struct {
	host string
	u    *unit.Unit
}

// TemplateRenderer used to render one unit.
type TemplateRenderer struct {
	t template.Template
}

// Deprecated: NewWithTemplate creates new TemplateRenderer instance for specified io.Reader unit.
// TemplateRenderer should be used to render unit for multiple hosts. For one host Unit() shorthand should be used.
// TemplateRenderer from reader cannot handle fragmented units.
func NewWithTemplate(u io.Reader) (*TemplateRenderer, error) {
	t, err := template.FromReader(u, "<render>", "<render>")
	if err != nil {
		return nil, err
	}
	return &TemplateRenderer{t: t}, nil
}

// NewTemplateFromStorage creates rendering template with specified storage and unit name
func NewTemplateFromStorage(s unitstorage.Storage, name string) (*TemplateRenderer, error) {
	t, err := template.FromStorage(s, name)
	if err != nil {
		return nil, err
	}
	return &TemplateRenderer{t: t}, nil
}

// Render produces Result for one hostpb.HostInfo.
func (r *TemplateRenderer) Render(hi *hostpb.HostInfo) (*Result, error) {
	rv := &Result{
		host: hi.Hostname,
	}
	result, err := r.t.Render(hi)
	if err != nil {
		return rv, err
	}
	rv.u = result
	return rv, nil
}

// Deprecated: Unit is a shorthand to render one unit with one hostpb.HostInfo.
// Rendering unit from reader cannot handle fragmented units.
func Unit(u io.Reader, hi *hostpb.HostInfo) (*Result, error) {
	r, err := NewWithTemplate(u)
	if err != nil {
		return nil, err
	}
	return r.Render(hi)
}

// UnitFromStorage is a shorthand to render one unit from specified storage with one hostpb.HostInfo.
func UnitFromStorage(s unitstorage.Storage, name string, hi *hostpb.HostInfo) (*Result, error) {
	r, err := NewTemplateFromStorage(s, name)
	if err != nil {
		return nil, err
	}
	return r.Render(hi)
}

// Pretty outputs human-readable unit representation as string.
func (r *Result) Pretty() (string, error) {
	return r.u.Prettify()
}

// RevMeta returns resulting hostpb.RevisionMeta.
func (r *Result) RevMeta() *hostpb.RevisionMeta {
	return r.u.RevisionMeta()
}

// SlotMeta returns resulting hostpb.SlotMeta.
func (r *Result) SlotMeta() *hostpb.SlotMeta {
	return r.u.SlotMeta()
}

// Spec returns resulting unit spec as proto.Message. This should be interpreted as one of (hostpb.PackageSetSpec,
// hostpb.SystemServiceSpec, hostpb.TimerJobSpec, hostpb.PortoDaemon). Unit kind can be found in SlotMeta().Kind field.
func (r *Result) Spec() proto.Message {
	return r.u.Spec()
}

// FQDN returns resulting host fqdn
func (r *Result) FQDN() string {
	return r.host
}

// Digest returns SHA1 digest computed for unit.
func (r *Result) Digest() string {
	return r.u.ID()
}

// Name return resulting unit name.
func (r *Result) Name() string {
	return r.u.Name()
}

// NewLocalRenderStorage creates minimal storage instance suitable for rendering purposes.
func NewLocalRenderStorage(unitPath string) (unitstorage.Storage, error) {
	absPath, err := filepath.Abs(unitPath)
	if err != nil {
		return nil, err
	}
	unitsDir := path.Dir(absPath)
	return unitstorage.NewDefaultFSStorage(unitstorage.NewLocalFS(), path.Dir(unitsDir), []string{unitsDir})
}

func UnitNameFromPath(unitPath string) string {
	filename := path.Base(unitPath)
	fileExt := path.Ext(filename)
	return filename[:len(filename)-len(fileExt)]
}
