package render

import (
	"bytes"
	"log"
	"os"
	"testing"

	"a.yandex-team.ru/infra/hostctl/pkg/unitstorage"
	"github.com/stretchr/testify/assert"
	"google.golang.org/protobuf/proto"

	hostpb "a.yandex-team.ru/infra/hostctl/proto"
)

// Example usage for rendering single unit
func ExampleUnit() {
	u, err := os.Open("unit.yaml")
	if err != nil {
		log.Fatalf("Cannot open unit.yaml: %v", err)
	}
	defer u.Close()
	rv, err := Unit(u, &hostpb.HostInfo{})
	if err != nil {
		log.Fatalf("Cannot render unit.yaml: %v", err)
	}
	rv.Digest()
}

// Example for multiple hosts rendering
func ExampleNewWithTemplate() {
	u, err := os.Open("unit.yaml")
	if err != nil {
		log.Fatalf("Cannot open unit.yaml: %v", err)
	}
	defer u.Close()
	renderer, err := NewWithTemplate(u)
	if err != nil {
		log.Fatalf("Cannot initialize TemplateRenderer: %v", err)
	}
	rv, err := renderer.Render(&hostpb.HostInfo{})
	if err != nil {
		log.Fatalf("Cannot render unit.yaml: %v", err)
	}
	rv.Digest()
}

// Example usage for rendering single unit
func ExampleUnitFromStorage() {
	s, err := NewLocalRenderStorage("unit.yaml")
	if err != nil {
		log.Fatalf("Cannot create render storage: %v", err)
	}
	unitName := UnitNameFromPath("unit.yaml")
	rv, err := UnitFromStorage(s, unitName, &hostpb.HostInfo{})
	if err != nil {
		log.Fatalf("Cannot render unit.yaml: %v", err)
	}
	rv.Digest()
}

// Example for multiple hosts rendering
func ExampleNewTemplateFromStorage() {
	s, err := NewLocalRenderStorage("unit.yaml")
	if err != nil {
		log.Fatalf("Cannot create render storage: %v", err)
	}
	unitName := UnitNameFromPath("unit.yaml")
	renderer, err := NewTemplateFromStorage(s, unitName)
	if err != nil {
		log.Fatalf("Cannot initialize TemplateRenderer: %v", err)
	}
	rv, err := renderer.Render(&hostpb.HostInfo{})
	if err != nil {
		log.Fatalf("Cannot render unit.yaml: %v", err)
	}
	rv.Digest()
}

func TestUnit(t *testing.T) {
	unitYaml := `vars:
  - name: stage
    match:
      - exp: "default()"
        # we aint ready to deal with optional ttyS1 instance
        val: "stable"
  - name: version
    match:
      - exp: "default()"
        val: "6240247"
---
meta:
  name: yandex-sol-rtc
  version: "{version}"
  kind: PackageSet
  annotations:
    stage: "{stage}"
    reconf-juggler: |+
      generate_aggregates: false
spec:
  packages:
    - name: yandex-sol-rtc
      version: "{version}"
`
	unitReader := bytes.NewReader([]byte(unitYaml))
	hi := &hostpb.HostInfo{
		Hostname:      "sas1-4920.search.yandex.net",
		KernelRelease: "4.19.172-41",
	}
	result, err := Unit(unitReader, hi)
	assert.NoError(t, err)
	assert.Equal(t, "sas1-4920.search.yandex.net", result.FQDN())
	assert.Equal(t, "yandex-sol-rtc", result.Name())
	assert.Equal(t, "PackageSet", result.RevMeta().Kind)
	assert.Equal(t, "6240247", result.RevMeta().Version)
	assert.Equal(t, "3d855da7b48406fc2df25abf1606bf6330951bee", result.Digest())
	assert.True(t, proto.Equal(result.Spec(), &hostpb.PackageSetSpec{
		Packages: []*hostpb.SystemPackage{
			{
				Name:    "yandex-sol-rtc",
				Version: "6240247",
			},
		},
		Files: nil,
	}))
}

func TestNewWithTemplate(t *testing.T) {
	unitYaml := `vars:
  - name: stage
    match:
      - exp: "default()"
        # we aint ready to deal with optional ttyS1 instance
        val: "stable"
  - name: version
    match:
      - exp: "default()"
        val: "6240247"
---
meta:
  name: yandex-sol-rtc
  version: "{version}"
  kind: PackageSet
  annotations:
    stage: "{stage}"
    reconf-juggler: |+
      generate_aggregates: false
spec:
  packages:
    - name: yandex-sol-rtc
      version: "{version}"
`
	unitReader := bytes.NewReader([]byte(unitYaml))
	hi := &hostpb.HostInfo{
		Hostname:      "sas1-4920.search.yandex.net",
		KernelRelease: "4.19.172-41",
	}
	renderer, err := NewWithTemplate(unitReader)
	assert.NoError(t, err)
	result, err := renderer.Render(hi)
	assert.NoError(t, err)
	assert.Equal(t, "sas1-4920.search.yandex.net", result.FQDN())
	assert.Equal(t, "yandex-sol-rtc", result.Name())
	assert.Equal(t, "PackageSet", result.RevMeta().Kind)
	assert.Equal(t, "6240247", result.RevMeta().Version)
	assert.Equal(t, "3d855da7b48406fc2df25abf1606bf6330951bee", result.Digest())
	assert.True(t, proto.Equal(result.Spec(), &hostpb.PackageSetSpec{
		Packages: []*hostpb.SystemPackage{
			{
				Name:    "yandex-sol-rtc",
				Version: "6240247",
			},
		},
		Files: nil,
	}))
}

func TestUnitFromStorage(t *testing.T) {
	unitYaml := `vars:
  - name: stage
    match:
      - exp: "default()"
        # we aint ready to deal with optional ttyS1 instance
        val: "stable"
  - name: version
    match:
      - exp: "default()"
        val: "6240247"
---
meta:
  name: yandex-sol-rtc
  version: "{version}"
  kind: PackageSet
  annotations:
    stage: "{stage}"
    reconf-juggler: |+
      generate_aggregates: false
spec:
  packages:
    - name: yandex-sol-rtc
      version: "{version}"
`
	fs := unitstorage.NewMemFS([]*unitstorage.MemFile{
		{
			Path:    "/home/test/yandex-sol-rtc.yaml",
			Content: unitYaml,
		},
	})
	s, err := unitstorage.NewDefaultFSStorage(fs, "/home", []string{"/home/test"})
	assert.NoError(t, err)
	hi := &hostpb.HostInfo{
		Hostname:      "sas1-4920.search.yandex.net",
		KernelRelease: "4.19.172-41",
	}
	result, err := UnitFromStorage(s, "yandex-sol-rtc", hi)
	assert.NoError(t, err)
	assert.Equal(t, "sas1-4920.search.yandex.net", result.FQDN())
	assert.Equal(t, "yandex-sol-rtc", result.Name())
	assert.Equal(t, "PackageSet", result.RevMeta().Kind)
	assert.Equal(t, "6240247", result.RevMeta().Version)
	assert.Equal(t, "3d855da7b48406fc2df25abf1606bf6330951bee", result.Digest())
	assert.True(t, proto.Equal(result.Spec(), &hostpb.PackageSetSpec{
		Packages: []*hostpb.SystemPackage{
			{
				Name:    "yandex-sol-rtc",
				Version: "6240247",
			},
		},
		Files: nil,
	}))
}

func TestNewTemplateFromStorage(t *testing.T) {
	unitYaml := `vars:
  - name: stage
    match:
      - exp: "default()"
        # we aint ready to deal with optional ttyS1 instance
        val: "stable"
  - name: version
    match:
      - exp: "default()"
        val: "6240247"
---
meta:
  name: yandex-sol-rtc
  version: "{version}"
  kind: PackageSet
  annotations:
    stage: "{stage}"
    reconf-juggler: |+
      generate_aggregates: false
spec:
  packages:
    - name: yandex-sol-rtc
      version: "{version}"
`
	fs := unitstorage.NewMemFS([]*unitstorage.MemFile{
		{
			Path:    "/home/test/yandex-sol-rtc.yaml",
			Content: unitYaml,
		},
	})
	s, err := unitstorage.NewDefaultFSStorage(fs, "/home", []string{"/home/test"})
	assert.NoError(t, err)
	hi := &hostpb.HostInfo{
		Hostname:      "sas1-4920.search.yandex.net",
		KernelRelease: "4.19.172-41",
	}
	renderer, err := NewTemplateFromStorage(s, "yandex-sol-rtc")
	assert.NoError(t, err)
	result, err := renderer.Render(hi)
	assert.NoError(t, err)
	assert.Equal(t, "sas1-4920.search.yandex.net", result.FQDN())
	assert.Equal(t, "yandex-sol-rtc", result.Name())
	assert.Equal(t, "PackageSet", result.RevMeta().Kind)
	assert.Equal(t, "6240247", result.RevMeta().Version)
	assert.Equal(t, "3d855da7b48406fc2df25abf1606bf6330951bee", result.Digest())
	assert.True(t, proto.Equal(result.Spec(), &hostpb.PackageSetSpec{
		Packages: []*hostpb.SystemPackage{
			{
				Name:    "yandex-sol-rtc",
				Version: "6240247",
			},
		},
		Files: nil,
	}))
}
