package rtcutil

import (
	"bytes"
	"fmt"
	"io"
	"io/ioutil"
	"strings"
	"sync"

	"a.yandex-team.ru/infra/hostctl/internal/slot"
	unit2 "a.yandex-team.ru/infra/hostctl/internal/unit"
	"a.yandex-team.ru/infra/hostctl/pkg/unitstorage"

	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/infra/hmserver/pkg/reporter/client"
	"a.yandex-team.ru/infra/hostctl/pkg/render"
	hostpb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/slices"
)

const (
	defaultThreads = 10
)

type clusterRendererClient interface {
	GetHostsCount() (int, error)
	GetHosts() (client.HostsReader, error)
}

// ProgressTick function to update rendering progress.
// The function will be called in each goroutine after unit rendering completes.
// Total number of hosts will be passed to function on each iteration.
// Can be used in progress bars in user interfaces.
type ProgressTick func(int)

// ClusterRendererParams contains tunable parameters for renderer.
// maxThreads - max number of concurrent goroutines to start for rendering.
// progressTick - function to be called after each render step complete.
type ClusterRendererParams struct {
	maxThreads   int
	semChan      chan bool
	progressTick ProgressTick
}

// ClusterRenderer represents an instance of renderer for specific RTC cluster.
type ClusterRenderer struct {
	params     ClusterRendererParams
	client     clusterRendererClient
	renderer   *render.TemplateRenderer
	reduceDone chan bool
	resultChan chan renderResult
	result     []*ClusterResult
	resultMap  map[string]*ClusterResult
	tasksWg    sync.WaitGroup
	hostsCount int
}

// ClusterResult represents rendering result of the unit.
type ClusterResult struct {
	name     string
	revMeta  *hostpb.RevisionMeta
	slotMeta *hostpb.SlotMeta
	spec     proto.Message
	err      error
	hosts    []string
	digest   string
}

type renderResult struct {
	result *render.Result
	err    error
}

// NewClusterRendererParams creates an instance of ClusterRendererParams with default values (noop ProgressTick and 10 threads).
func NewClusterRendererParams() *ClusterRendererParams {
	return &ClusterRendererParams{
		maxThreads:   defaultThreads,
		semChan:      make(chan bool, defaultThreads),
		progressTick: func(_ int) {},
	}
}

// WithMaxThreads sets maxThreads value.
func (p *ClusterRendererParams) WithMaxThreads(maxThreads int) *ClusterRendererParams {
	p.maxThreads = maxThreads
	p.semChan = make(chan bool, maxThreads)
	return p
}

// WithProgressTick sets progress tick function.
func (p *ClusterRendererParams) WithProgressTick(tickFunc ProgressTick) *ClusterRendererParams {
	p.progressTick = tickFunc
	return p
}

// NewClusterRenderer creates ClusterRenderer instance with default params (reasonable for most use cases).
func NewClusterRenderer(cluster string) (*ClusterRenderer, error) {
	return NewClusterRendererWithParams(cluster, *NewClusterRendererParams())
}

// NewClusterRendererWithParams creates ClusterRenderer with specific ClusterRendererParams.
func NewClusterRendererWithParams(cluster string, params ClusterRendererParams) (*ClusterRenderer, error) {
	if ok := slices.ContainsString(rtcClusters, cluster); !ok {
		return nil, fmt.Errorf("unknown rtc cluster %s, should be oneof [%s]", cluster, strings.Join(rtcClusters, ", "))
	}
	return &ClusterRenderer{
		params:     params,
		client:     client.NewReporter(reporterAddrs[cluster]),
		reduceDone: make(chan bool),
		resultChan: make(chan renderResult, 200),
		result:     make([]*ClusterResult, 0),
		resultMap:  make(map[string]*ClusterResult),
	}, nil
}

// lock semaphore and wait group of concurrent render tasks
func (r *ClusterRenderer) lock() {
	r.tasksWg.Add(1)
	r.params.semChan <- true
}

// unlock semaphore and wait group of concurrent render tasks
func (r *ClusterRenderer) unlock() {
	<-r.params.semChan
	r.tasksWg.Done()
}

// collectResults collects results from concurrent render tasks via channel
func (r *ClusterRenderer) collectResults() {
	for result := range r.resultChan {
		var id string
		digest := ""
		if result.err != nil {
			id = result.err.Error()
		} else {
			stage := slot.MetaFromProto(result.result.SlotMeta()).Stage()
			ver := slot.RevisionMetaFromProto(result.result.RevMeta()).GetVersion()
			digest = result.result.Digest()
			id = fmt.Sprintf("%s:%s:%s", stage, ver, digest)
		}
		if _, ok := r.resultMap[id]; ok {
			r.resultMap[id].hosts = append(r.resultMap[id].hosts, result.result.FQDN())
		} else {
			var rv *ClusterResult
			// On rendering with error we can have empty result
			if result.err != nil {
				rv = &ClusterResult{
					err:    result.err,
					hosts:  []string{result.result.FQDN()},
					digest: digest,
				}
			} else {
				rv = &ClusterResult{
					name:     result.result.Name(),
					revMeta:  result.result.RevMeta(),
					slotMeta: result.result.SlotMeta(),
					spec:     result.result.Spec(),
					hosts:    []string{result.result.FQDN()},
					digest:   digest,
				}
			}
			r.result = append(r.result, rv)
			r.resultMap[id] = rv
		}
	}
	r.reduceDone <- true
}

// emitRenderTask emits concurrent render task for HostInfo according to params
func (r *ClusterRenderer) emitRenderTask(hi *hostpb.HostInfo) {
	r.lock()
	go r.runRenderTask(hi)
}

// runRenderTask renders template using provided HostInfo
func (r *ClusterRenderer) runRenderTask(hi *hostpb.HostInfo) {
	rv, err := r.renderer.Render(hi)
	r.params.progressTick(r.hostsCount)
	r.resultChan <- renderResult{
		result: rv,
		err:    err,
	}
	r.unlock()
}

// Deprecated: Unit renders unit on all hosts of specified RTC cluster. During rendering results are grouped using digest or error value.
// Unit cannot handle fragmented units.
func (r *ClusterRenderer) Unit(u io.Reader) ([]*ClusterResult, error) {
	unitBuf, err := ioutil.ReadAll(u)
	if err != nil {
		return nil, err
	}
	r.renderer, err = render.NewWithTemplate(bytes.NewReader(unitBuf))
	if err != nil {
		return nil, err
	}
	hostsCount, err := r.client.GetHostsCount()
	if err != nil {
		return nil, err
	}
	r.hostsCount = hostsCount
	reader, err := r.client.GetHosts()
	if err != nil {
		return nil, err
	}
	go r.collectResults()
	err = r.emitRenderTasks(reader)
	if err != nil {
		return nil, err
	}

	<-r.reduceDone
	return r.result, nil
}

// emitRenderTasks reads HostInfo from reader and emits render task for each HostInfo.
func (r *ClusterRenderer) emitRenderTasks(reader client.HostsReader) error {
	defer func() {
		r.tasksWg.Wait()
		close(r.resultChan)
	}()
	for {
		hi, next, err := reader.Read()
		if err != nil {
			return err
		}
		if !next {
			break
		}
		r.emitRenderTask(hi)
	}
	return nil
}

// UnitFromStorage renders unit on all hosts of specified RTC cluster. During rendering results are grouped using digest or error value.
func (r *ClusterRenderer) UnitFromStorage(s unitstorage.Storage, name string) ([]*ClusterResult, error) {
	var err error
	r.renderer, err = render.NewTemplateFromStorage(s, name)
	if err != nil {
		return nil, err
	}
	hostsCount, err := r.client.GetHostsCount()
	if err != nil {
		return nil, err
	}
	r.hostsCount = hostsCount
	reader, err := r.client.GetHosts()
	if err != nil {
		return nil, err
	}
	go r.collectResults()
	err = r.emitRenderTasks(reader)
	if err != nil {
		return nil, err
	}

	<-r.reduceDone
	return r.result, nil
}

// Pretty outputs human-readable unit representation as string.
func (r *ClusterResult) Pretty() (string, error) {
	return unit2.Prettify(r.name, r.digest, r.revMeta, r.slotMeta, r.spec)
}

// RevMeta returns resulting hostpb.RevisionMeta.
func (r *ClusterResult) RevMeta() *hostpb.RevisionMeta {
	return r.revMeta
}

// SlotMeta returns resulting hostpb.SlotMeta.
func (r *ClusterResult) SlotMeta() *hostpb.SlotMeta {
	return r.slotMeta
}

// Spec returns resulting unit spec as proto.Message. This should be interpreted as one of (hostpb.PackageSetSpec,
// hostpb.SystemServiceSpec, hostpb.TimerJobSpec, hostpb.PortoDaemon). Unit kind can be found in SlotMeta().Kind field.
func (r *ClusterResult) Spec() proto.Message {
	return r.spec
}

// Error return error for result or nil if there was no errors.
func (r *ClusterResult) Error() error {
	return r.err
}

// Hosts returns a slice of host fqdns rendered same unit.
func (r *ClusterResult) Hosts() []string {
	return r.hosts
}

// Digest returns SHA1 digest computed for unit.
func (r *ClusterResult) Digest() string {
	return r.digest
}

// Name return resulting unit name.
func (r *ClusterResult) Name() string {
	return r.name
}
