package rtcutil

import (
	"bytes"
	"errors"
	"testing"

	"a.yandex-team.ru/infra/hostctl/pkg/unitstorage"
	"github.com/stretchr/testify/assert"

	"a.yandex-team.ru/infra/hmserver/pkg/reporter/client"
	hostpb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/slices"
)

type mockClusterClient struct {
	info map[string]*hostpb.HostInfo
}

type mockClusterClientReader struct {
	client *mockClusterClient
	skip   []string
}

func (m *mockClusterClientReader) Read() (*hostpb.HostInfo, bool, error) {
	if len(m.client.info) == len(m.skip) {
		m.skip = append(m.skip, "null")
		return nil, false, nil
	}
	for k, v := range m.client.info {
		if slices.ContainsString(m.skip, k) {
			continue
		} else {
			m.skip = append(m.skip, k)
			return v, true, nil
		}
	}
	return nil, false, errors.New("undefined behaviour")
}

func (m *mockClusterClientReader) Close() error {
	return nil
}

func (m *mockClusterClient) GetHostsCount() (int, error) {
	return len(m.info), nil
}

func (m *mockClusterClient) GetHosts() (client.HostsReader, error) {
	return &mockClusterClientReader{
		client: m,
		skip:   make([]string, 0),
	}, nil
}

func TestClusterRenderer_Render(t *testing.T) {
	unitYaml := `vars:
  - name: stage
    match:
      - exp: "default()"
        # we aint ready to deal with optional ttyS1 instance
        val: "stable"
  - name: version
    match:
      - exp: "default()"
        val: "6240247"
---
meta:
  name: yandex-sol-rtc
  version: "{version}"
  kind: PackageSet
  annotations:
    stage: "{stage}"
    reconf-juggler: |+
      generate_aggregates: false
spec:
  packages:
    - name: yandex-sol-rtc
      version: "{version}"
`
	unitReader := bytes.NewReader([]byte(unitYaml))
	hi1 := &hostpb.HostInfo{
		Hostname:      "sas1-4920.search.yandex.net",
		KernelRelease: "4.19.172-41",
	}
	hi2 := &hostpb.HostInfo{
		Hostname:      "sas1-4921.search.yandex.net",
		KernelRelease: "4.19.172-41",
	}
	c := &mockClusterClient{info: map[string]*hostpb.HostInfo{"sas1-4920.search.yandex.net": hi1, "sas1-4921.search.yandex.net": hi2}}
	renderer, err := NewClusterRenderer("PRESTABLE")
	assert.NoError(t, err)
	renderer.client = c
	result, err := renderer.Unit(unitReader)
	assert.NoError(t, err)
	assert.Equal(t, 2, len(result[0].Hosts()))
	_, err = renderer.Unit(unitReader)
	assert.Error(t, err)
}

func TestClusterRenderer_RenderFromStorage(t *testing.T) {
	unitYaml := `vars:
  - name: stage
    match:
      - exp: "default()"
        # we aint ready to deal with optional ttyS1 instance
        val: "stable"
  - name: version
    match:
      - exp: "default()"
        val: "6240247"
---
meta:
  name: yandex-sol-rtc
  version: "{version}"
  kind: PackageSet
  annotations:
    stage: "{stage}"
    reconf-juggler: |+
      generate_aggregates: false
spec:
  packages:
    - name: yandex-sol-rtc
      version: "{version}"
`
	fs := unitstorage.NewMemFS([]*unitstorage.MemFile{
		{
			Path:    "/home/test/yandex-sol-rtc.yaml",
			Content: unitYaml,
		},
	})
	s, err := unitstorage.NewDefaultFSStorage(fs, "/home", []string{"/home/test"})
	assert.NoError(t, err)
	hi1 := &hostpb.HostInfo{
		Hostname:      "sas1-4920.search.yandex.net",
		KernelRelease: "4.19.172-41",
	}
	hi2 := &hostpb.HostInfo{
		Hostname:      "sas1-4921.search.yandex.net",
		KernelRelease: "4.19.172-41",
	}
	c := &mockClusterClient{info: map[string]*hostpb.HostInfo{"sas1-4920.search.yandex.net": hi1, "sas1-4921.search.yandex.net": hi2}}
	renderer, err := NewClusterRenderer("PRESTABLE")
	assert.NoError(t, err)
	renderer.client = c
	result, err := renderer.UnitFromStorage(s, "yandex-sol-rtc")
	assert.NoError(t, err)
	assert.Equal(t, 2, len(result[0].Hosts()))
	fs = unitstorage.NewMemFS([]*unitstorage.MemFile{
		{
			Path:    "/home/test/yandex-sol-rtc.yaml",
			Content: "",
		},
	})
	s, err = unitstorage.NewDefaultFSStorage(fs, "/home", []string{"/home/test"})
	assert.NoError(t, err)
	_, err = renderer.UnitFromStorage(s, "yandex-sol-rtc")
	assert.Error(t, err)
}
