package rtcutil

import (
	"bytes"
	"fmt"
	"io"
	"testing"

	"a.yandex-team.ru/infra/hostctl/pkg/unitstorage"
	"github.com/stretchr/testify/assert"
	"google.golang.org/protobuf/proto"

	hostpb "a.yandex-team.ru/infra/hostctl/proto"
)

func TestNewHostRendererForCluster(t *testing.T) {
	r, err := NewHostRendererForCluster("PRESTABLE")
	assert.NoError(t, err)
	assert.NotNil(t, r.client)
	r, err = NewHostRendererForCluster("SAS")
	assert.NoError(t, err)
	assert.NotNil(t, r.client)
	r, err = NewHostRendererForCluster("VLA")
	assert.NoError(t, err)
	assert.NotNil(t, r.client)
	r, err = NewHostRendererForCluster("MAN")
	assert.NoError(t, err)
	assert.NotNil(t, r.client)
	r, err = NewHostRendererForCluster("MSK")
	assert.NoError(t, err)
	assert.NotNil(t, r.client)
}

type mockHostRendererClient struct {
	info map[string]*hostpb.HostInfo
}

func (c *mockHostRendererClient) GetHost(fqdn string) (*hostpb.HostInfo, error) {
	if hi, ok := c.info[fqdn]; ok {
		return hi, nil
	} else {
		return nil, fmt.Errorf("host %s not found", fqdn)
	}
}

func TestHostRenderer_Render(t *testing.T) {
	unitYaml := `vars:
  - name: stage
    match:
      - exp: "default()"
        # we aint ready to deal with optional ttyS1 instance
        val: "stable"
  - name: version
    match:
      - exp: "default()"
        val: "6240247"
---
meta:
  name: yandex-sol-rtc
  version: "{version}"
  kind: PackageSet
  annotations:
    stage: "{stage}"
    reconf-juggler: |+
      generate_aggregates: false
spec:
  packages:
    - name: yandex-sol-rtc
      version: "{version}"
`
	unitReader := bytes.NewReader([]byte(unitYaml))
	hi := &hostpb.HostInfo{
		Hostname:      "sas1-4920.search.yandex.net",
		KernelRelease: "4.19.172-41",
	}
	client := &mockHostRendererClient{info: map[string]*hostpb.HostInfo{"sas1-4920.search.yandex.net": hi}}
	renderer, err := NewHostRendererForCluster("PRESTABLE")
	assert.NoError(t, err)
	renderer.client = client
	result, err := renderer.UnitOn(unitReader, "sas1-4920.search.yandex.net")
	assert.NoError(t, err)
	assert.Equal(t, "sas1-4920.search.yandex.net", result.FQDN())
	assert.Equal(t, "yandex-sol-rtc", result.Name())
	assert.Equal(t, "PackageSet", result.RevMeta().Kind)
	assert.Equal(t, "6240247", result.RevMeta().Version)
	assert.Equal(t, "3d855da7b48406fc2df25abf1606bf6330951bee", result.Digest())
	assert.True(t, proto.Equal(result.Spec(), &hostpb.PackageSetSpec{
		Packages: []*hostpb.SystemPackage{
			{
				Name:    "yandex-sol-rtc",
				Version: "6240247",
			},
		},
		Files: nil,
	}))
	_, _ = unitReader.Seek(0, io.SeekStart)
	_, err = renderer.UnitOn(unitReader, "non-existing-host")
	assert.Error(t, err)
}

func TestHostRenderer_RenderFromStorage(t *testing.T) {
	unitYaml := `vars:
  - name: stage
    match:
      - exp: "default()"
        # we aint ready to deal with optional ttyS1 instance
        val: "stable"
  - name: version
    match:
      - exp: "default()"
        val: "6240247"
---
meta:
  name: yandex-sol-rtc
  version: "{version}"
  kind: PackageSet
  annotations:
    stage: "{stage}"
    reconf-juggler: |+
      generate_aggregates: false
spec:
  packages:
    - name: yandex-sol-rtc
      version: "{version}"
`
	fs := unitstorage.NewMemFS([]*unitstorage.MemFile{
		{
			Path:    "/home/test/yandex-sol-rtc.yaml",
			Content: unitYaml,
		},
	})
	s, err := unitstorage.NewDefaultFSStorage(fs, "/home", []string{"/home/test"})
	assert.NoError(t, err)
	hi := &hostpb.HostInfo{
		Hostname:      "sas1-4920.search.yandex.net",
		KernelRelease: "4.19.172-41",
	}
	client := &mockHostRendererClient{info: map[string]*hostpb.HostInfo{"sas1-4920.search.yandex.net": hi}}
	renderer, err := NewHostRendererForCluster("PRESTABLE")
	assert.NoError(t, err)
	renderer.client = client
	result, err := renderer.UnitFromStorage(s, "yandex-sol-rtc", "sas1-4920.search.yandex.net")
	assert.NoError(t, err)
	assert.Equal(t, "sas1-4920.search.yandex.net", result.FQDN())
	assert.Equal(t, "yandex-sol-rtc", result.Name())
	assert.Equal(t, "PackageSet", result.RevMeta().Kind)
	assert.Equal(t, "6240247", result.RevMeta().Version)
	assert.Equal(t, "3d855da7b48406fc2df25abf1606bf6330951bee", result.Digest())
	assert.True(t, proto.Equal(result.Spec(), &hostpb.PackageSetSpec{
		Packages: []*hostpb.SystemPackage{
			{
				Name:    "yandex-sol-rtc",
				Version: "6240247",
			},
		},
		Files: nil,
	}))
	_, err = renderer.UnitFromStorage(s, "yandex-sol-rtc", "non-existing-host")
	assert.Error(t, err)
}
