package unitstorage

import (
	"fmt"
	"io"
	"io/fs"
)

// CtxReader defines interface used to read contents of "context" document.
// "context" document contains variables that are evaluated using cel-go engine.
type CtxReader io.Reader

// SpecReader defines interface used to read contents of spec document.
// spec document contains fields `meta` and `spec` so we need an io.Seeker
// to read the document twice: when parsing meta to pb.ObjectMeta
// and when parsing spec to corresponding spec struct.
type SpecReader interface {
	io.Reader
	io.Seeker
}

// Document defines interface representing template document read from filesystem
// or another source.
type Document interface {
	// Stringer used for pretty printing
	fmt.Stringer
	// CtxReader returns context part reader
	CtxReader() CtxReader
	// SpecReader returns spec part reader
	SpecReader() SpecReader
	// Path returns fs path from which file was read
	Path() string
	// Repo returns repo name from which file was read
	Repo() string
}

// File represents raw file from storage
type File struct {
	Repo   string
	Path   string
	Reader io.ReadCloser
}

// Storage encapsulates unit lookup and file access
type Storage interface {
	// OpenFile opens File by name from underlying filesystem according to overrides rules
	OpenFile(string) (*File, error)
	// DiscoverUnits returns list of all files that are look like units
	DiscoverUnits() ([]string, error)
}

// FS defines used filesystem interface
// Special errors should be checked and wrapped against fs.Err* instead of os.err*
type FS interface {
	fs.FS
	fs.ReadDirFS
	fs.StatFS
}
