package unitstorage

import (
	"fmt"
	"path"
	"strings"

	"a.yandex-team.ru/library/go/slices"
)

const (
	repoLookupUnits     = "units.d"
	repoLookupDaemons   = "porto-daemons.d"
	repoSystemOverrides = "<system-overrides>"
	repoUserOverrides   = "<user-overrides>"
)

var (
	overrideRepos = []string{repoSystemOverrides, repoUserOverrides}
)

// repoPath stores metadata about directory with unit files
type repoPath struct {
	// repoRoot points to repo root, usually /var/lib/ya-salt/repo/current
	repoRoot string
	// repo contains repo name, for path /var/lib/ya-salt/repo/current/core/units.d it will be "core"
	repo string
	// lookupPath contains path for unit files lookup
	lookupPath string
}

// DefaultFSStorage is default implementation for Storage interface
type DefaultFSStorage struct {
	lookupPaths []*repoPath
	fs          FS
}

// NewDefaultFSStorage constructs DefaultFSStorage instance for underlying FS
// DefaultFSStorage will look up units in repo dirs, system overrides and then in user overrides
func NewDefaultFSStorage(f FS, repoRoot string, explicitUserOverrides []string) (*DefaultFSStorage, error) {
	lookupPaths, err := discoverPaths(f, repoRoot, explicitUserOverrides)
	if err != nil {
		return nil, err
	}
	return &DefaultFSStorage{
		lookupPaths: lookupPaths,
		fs:          f,
	}, nil
}

// OpenFile opens File honouring system and user overrides
func (s *DefaultFSStorage) OpenFile(path string) (*File, error) {
	fsPath, repo, err := s.lookupFile(path)
	if err != nil {
		return nil, err
	}
	f, err := s.fs.Open(fsPath)
	if err != nil {
		return nil, fmt.Errorf("failed to open file from storage: %v", err)
	}
	return &File{
		Repo:   repo,
		Path:   fsPath,
		Reader: f,
	}, nil
}

// DiscoverUnits discovers all possible units in storage (used for ManageAll hostctl mode)
func (s *DefaultFSStorage) DiscoverUnits() ([]string, error) {
	units := make(map[string]struct{})
	for _, dir := range s.lookupPaths {
		entries, err := s.fs.ReadDir(dir.lookupPath)
		if err != nil {
			return nil, err
		}
		for _, dent := range entries {
			name := dent.Name()
			if strings.HasSuffix(name, ".yaml") {
				units[name[0:len(name)-5]] = struct{}{}
			}
		}
	}
	var rv []string
	for k := range units {
		rv = append(rv, k)
	}
	return rv, nil
}

// lookupFile looks for path in storage honouring system and user overrides
func (s *DefaultFSStorage) lookupFile(p string) (string, string, error) {
	var foundPath, foundRepo string
	for _, candidate := range s.lookupPaths {
		candidatePath := path.Join(candidate.lookupPath, p)
		_, err := s.fs.Stat(candidatePath)
		if err == nil {
			// TODO: check that units has diff for conflict
			if foundRepo != "" && candidate.repo != foundRepo && !slices.ContainsString(overrideRepos, candidate.repo) {
				return "", "", fmt.Errorf("conflict looking for \"%s\": repos %s and %s both contains same path", p, foundRepo, candidate.repo)
			}
			foundPath = candidatePath
			foundRepo = candidate.repo
		}
	}
	if foundPath == "" {
		paths := make([]string, 0, len(s.lookupPaths))
		for _, p := range s.lookupPaths {
			paths = append(paths, p.lookupPath)
		}
		return "", "", fmt.Errorf("file %s not found (looked in %s)", p, strings.Join(paths, ", "))
	}
	return foundPath, foundRepo, nil
}

// discoverPaths discovers directories that could contain unit files
func discoverPaths(f FS, repoRoot string, explicitUserOverrides []string) ([]*repoPath, error) {
	var rv []*repoPath
	if repoRoot != "" {
		repoPaths, err := findRepos(f, repoRoot)
		if err != nil {
			return nil, err
		}
		rv = append(rv, repoPaths...)
		rv = append(rv, systemOverrides(f)...)
	}
	rv = append(rv, userOverrides(f, explicitUserOverrides)...)
	return rv, nil
}

// systemOverrides checks if there are system override dirs exist and returns existing paths
func systemOverrides(f FS) []*repoPath {
	possiblePaths := []*repoPath{
		{
			repoRoot:   "/",
			repo:       repoSystemOverrides,
			lookupPath: "/etc/hostman/porto-daemons.d",
		},
		{
			repoRoot:   "/",
			repo:       repoSystemOverrides,
			lookupPath: "/etc/hostman/units.d",
		},
		{
			repoRoot:   "/",
			repo:       repoSystemOverrides,
			lookupPath: "/run/hostman/porto-daemons.d",
		},
		{
			repoRoot:   "/",
			repo:       repoSystemOverrides,
			lookupPath: "/run/hostman/units.d",
		},
	}
	var rv []*repoPath
	for _, candidate := range possiblePaths {
		stat, err := f.Stat(candidate.lookupPath)
		if err == nil && stat.IsDir() {
			rv = append(rv, candidate)
		}
	}
	return rv
}

// userOverrides checks if there are user override dirs exist and returns existing paths
func userOverrides(f FS, userOverrides []string) []*repoPath {
	var rv []*repoPath
	for _, p := range userOverrides {
		if stat, err := f.Stat(p); err == nil && stat.IsDir() {
			rv = append(rv, &repoPath{
				repoRoot:   "/",
				repo:       repoUserOverrides,
				lookupPath: p,
			})
		}
	}
	return rv
}

// findRepos discovers all directories under repoRoot that could contain units
// usually <repoRoot>/<repo>/{units.d,porto-daemons.d}
func findRepos(f FS, repoRoot string) ([]*repoPath, error) {
	var rv []*repoPath
	dentries, err := f.ReadDir(repoRoot)
	if err != nil {
		return nil, err
	}
	for _, dent := range dentries {
		if dent.IsDir() {
			dirs, err := findRepoUnitsPaths(f, repoRoot, dent.Name())
			if err != nil {
				return nil, err
			}
			rv = append(rv, dirs...)
		}
	}
	return rv, nil
}

// findRepoUnitsPaths checks if repoCandidate has units.d or porto-daemons.d subdir and returns existing
func findRepoUnitsPaths(f FS, repoRoot, repoCandidate string) ([]*repoPath, error) {
	var rv []*repoPath
	unitsPath := fmtRepoPath(repoRoot, repoCandidate, repoLookupUnits)
	if pathIsDir(f, unitsPath) {
		rv = append(rv, &repoPath{
			repoRoot:   repoRoot,
			repo:       repoCandidate,
			lookupPath: unitsPath,
		})
	}
	daemonsPath := fmtRepoPath(repoRoot, repoCandidate, repoLookupDaemons)
	if pathIsDir(f, daemonsPath) {
		rv = append(rv, &repoPath{
			repoRoot:   repoRoot,
			repo:       repoCandidate,
			lookupPath: daemonsPath,
		})
	}
	return rv, nil
}

// fmtRepoPath formats full path to repo from parts <repoRoot>/<repoCandidate>/<repoSubDir>
func fmtRepoPath(repoRoot, repoCandidate, repoSubDir string) string {
	return path.Join(repoRoot, repoCandidate, repoSubDir)
}

// pathIsDir check if path exists and is directory
func pathIsDir(f FS, path string) bool {
	stat, err := f.Stat(path)
	if err != nil {
		return false
	}
	if stat.IsDir() {
		return true
	}
	return false
}
