package unitstorage

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

func mockMemFS() FS {
	return NewMemFS([]*MemFile{
		{
			Path:    "/var/lib/ya-salt/repo/current/core/units.d/apt.yaml",
			Content: "apt repo",
			Dir:     false,
		},
		{
			Path:    "/etc/hostman/units.d/apt.yaml",
			Content: "apt etc",
			Dir:     false,
		},
		{
			Path:    "/var/lib/ya-salt/repo/current/sysdev/porto-daemons.d/yandex-hbf-agent.yaml",
			Content: "hbf repo",
			Dir:     false,
		},
		{
			Path:    "/home/loadbase/ssh.yaml",
			Content: "ssh loadbase",
			Dir:     false,
		},
	})
}

func TestPathIsDir(t *testing.T) {
	f := mockMemFS()
	assert.True(t, pathIsDir(f, "/var/lib/ya-salt/repo/current/core/units.d"))
	assert.False(t, pathIsDir(f, "/var/lib/ya-salt/repo/current/core/units.d/apt.yaml"))
	assert.False(t, pathIsDir(f, "/non-existing-path"))
}

func TestFmtRepoPath(t *testing.T) {
	assert.Equal(t,
		"/var/lib/ya-salt/repo/current/core/porto-daemons.d",
		fmtRepoPath("/var/lib/ya-salt/repo/current", "core", "porto-daemons.d"),
	)
}

func TestFindRepoUnitsPaths(t *testing.T) {
	f := mockMemFS()
	expected := []*repoPath{{
		repoRoot:   "/var/lib/ya-salt/repo/current",
		repo:       "core",
		lookupPath: "/var/lib/ya-salt/repo/current/core/units.d",
	}}
	paths, err := findRepoUnitsPaths(f, "/var/lib/ya-salt/repo/current", "core")
	assert.NoError(t, err)
	assert.Equal(t, expected, paths)
}

func TestFindRepos(t *testing.T) {
	f := mockMemFS()
	repos, err := findRepos(f, "/var/lib/ya-salt/repo/current")
	assert.NoError(t, err)
	assert.Equal(t, []*repoPath{
		{
			repoRoot:   "/var/lib/ya-salt/repo/current",
			repo:       "core",
			lookupPath: "/var/lib/ya-salt/repo/current/core/units.d",
		},
		{
			repoRoot:   "/var/lib/ya-salt/repo/current",
			repo:       "sysdev",
			lookupPath: "/var/lib/ya-salt/repo/current/sysdev/porto-daemons.d",
		},
	}, repos)
}

func TestUserOverrides(t *testing.T) {
	f := mockMemFS()
	overrides := userOverrides(f, []string{"/root", "/home/loadbase"})
	assert.Equal(t, []*repoPath{
		{
			repoRoot:   "/",
			repo:       repoUserOverrides,
			lookupPath: "/home/loadbase",
		},
	}, overrides)
}

func TestSystemOverrides(t *testing.T) {
	f := mockMemFS()
	overrides := systemOverrides(f)
	assert.Equal(t, []*repoPath{
		{
			repoRoot:   "/",
			repo:       repoSystemOverrides,
			lookupPath: "/etc/hostman/units.d",
		},
	}, overrides)
}

func TestDiscoverPaths(t *testing.T) {
	f := mockMemFS()
	paths, err := discoverPaths(f, "/var/lib/ya-salt/repo/current", []string{"/home/loadbase"})
	assert.NoError(t, err)
	assert.Equal(t, []*repoPath{
		{
			repoRoot:   "/var/lib/ya-salt/repo/current",
			repo:       "core",
			lookupPath: "/var/lib/ya-salt/repo/current/core/units.d",
		},
		{
			repoRoot:   "/var/lib/ya-salt/repo/current",
			repo:       "sysdev",
			lookupPath: "/var/lib/ya-salt/repo/current/sysdev/porto-daemons.d",
		},
		{
			repoRoot:   "/",
			repo:       repoSystemOverrides,
			lookupPath: "/etc/hostman/units.d",
		},
		{
			repoRoot:   "/",
			repo:       repoUserOverrides,
			lookupPath: "/home/loadbase",
		},
	}, paths)
}

func TestFSStorage_lookupFile(t *testing.T) {
	f := mockMemFS()
	s, err := NewDefaultFSStorage(f, "/var/lib/ya-salt/repo/current", []string{"/home/loadbase"})
	assert.NoError(t, err)
	path, repo, err := s.lookupFile("apt.yaml")
	assert.NoError(t, err)
	assert.Equal(t, "/etc/hostman/units.d/apt.yaml", path)
	assert.Equal(t, repoSystemOverrides, repo)
	path, repo, err = s.lookupFile("yandex-hbf-agent.yaml")
	assert.NoError(t, err)
	assert.Equal(t, "/var/lib/ya-salt/repo/current/sysdev/porto-daemons.d/yandex-hbf-agent.yaml", path)
	assert.Equal(t, "sysdev", repo)
	_, _, err = s.lookupFile("non-existent.yaml")
	assert.EqualError(t, err, "file non-existent.yaml not found (looked in /var/lib/ya-salt/repo/current/core/units.d, /var/lib/ya-salt/repo/current/sysdev/porto-daemons.d, /etc/hostman/units.d, /home/loadbase)")
}
