package unitstorage

import (
	"fmt"
	"io"
	"io/fs"
	"path"
	"strings"
	"time"
)

type mReadCloser struct {
	r io.Reader
}

func (r *mReadCloser) Read(p []byte) (int, error) {
	return r.r.Read(p)
}

func (r *mReadCloser) Close() error {
	return nil
}

type mSingleUnitStorage struct {
	name    string
	path    string
	repo    string
	content string
}

func (m *mSingleUnitStorage) OpenFile(s string) (*File, error) {
	if s == m.name {
		return &File{
			Repo:   m.repo,
			Path:   m.path,
			Reader: &mReadCloser{r: strings.NewReader(m.content)},
		}, nil
	}
	return nil, fmt.Errorf("%w: %s", fs.ErrNotExist, s)
}

func (m *mSingleUnitStorage) DiscoverUnits() ([]string, error) {
	return []string{m.name[:len(m.name)-5]}, nil
}

func NewSingleUnitStorage(name, path, repo, content string) Storage {
	return &mSingleUnitStorage{
		name:    name,
		path:    path,
		repo:    repo,
		content: content,
	}
}

// MemFS related code
// Using custom implementation of memory fs for testing purposes
// as fstest.MapFS forces "relative" paths to be used:
// "etc/hostman/units.d" instead of "/etc/hostman/units.d"
// using "relative" paths everywhere is a bit inconvenient

type MemFile struct {
	Path    string
	Content string
	Reader  io.Reader
	Dir     bool
}

func (m *MemFile) Type() fs.FileMode {
	return 0o644
}

func (m *MemFile) Info() (fs.FileInfo, error) {
	return m, nil
}

func (m *MemFile) Stat() (fs.FileInfo, error) {
	return m, nil
}

func (m *MemFile) Read(bytes []byte) (int, error) {
	if m.Reader == nil {
		m.Reader = strings.NewReader(m.Content)
	}
	return m.Reader.Read(bytes)
}

func (m *MemFile) Close() error {
	m.Reader = nil
	return nil
}

func (m *MemFile) Name() string {
	return path.Base(m.Path)
}

func (m *MemFile) Size() int64 {
	return int64(len(m.Content))
}

func (m *MemFile) Mode() fs.FileMode {
	return 0o644
}

func (m *MemFile) ModTime() time.Time {
	return time.Now()
}

func (m *MemFile) IsDir() bool {
	return m.Dir
}

func (m *MemFile) Sys() interface{} {
	return nil
}

type MemDir struct {
	files []*MemFile
	name  string
	path  string
}

type MemFS struct {
	Files map[string]*MemFile
	dirs  map[string]*MemDir
}

func memfsDir(dirs map[string]*MemDir, files map[string]*MemFile, dirPath string, childDir *MemFile) {
	if parentDir, ok := dirs[dirPath]; !ok {
		dirName := path.Base(dirPath)
		dirs[dirPath] = &MemDir{
			files: nil,
			name:  dirName,
			path:  dirPath,
		}
		if childDir != nil {
			dirs[dirPath].files = append(dirs[dirPath].files, childDir)
		}
		files[dirPath] = &MemFile{
			Path: dirPath,
			Dir:  true,
		}
		parent := path.Dir(dirPath)
		if parent != dirPath {
			memfsDir(dirs, files, parent, files[dirPath])
		}
	} else {
		if childDir != nil {
			parentDir.files = append(parentDir.files, childDir)
		}
	}
}

func NewMemFS(files []*MemFile) FS {
	dirs := make(map[string]*MemDir)
	filesMap := make(map[string]*MemFile)
	for _, f := range files {
		filesMap[f.Path] = f
		dir := path.Dir(f.Path)
		memfsDir(dirs, filesMap, dir, nil)
		dirs[dir].files = append(dirs[dir].files, f)
	}
	return &MemFS{
		Files: filesMap,
		dirs:  dirs,
	}
}

func (m *MemFS) Open(name string) (fs.File, error) {
	if f, ok := m.Files[name]; ok {
		return f, nil
	}
	return nil, &fs.PathError{Path: name, Err: fs.ErrNotExist, Op: "open"}
}

func (m *MemFS) ReadDir(name string) ([]fs.DirEntry, error) {
	if dir, ok := m.dirs[name]; ok {
		files := make([]fs.DirEntry, len(dir.files))
		for i := range dir.files {
			files[i] = dir.files[i]
		}
		return files, nil
	}
	return nil, &fs.PathError{Path: name, Err: fs.ErrNotExist, Op: "readdir"}
}

func (m *MemFS) Stat(name string) (fs.FileInfo, error) {
	if file, ok := m.Files[name]; ok {
		return file, nil
	}
	return nil, &fs.PathError{Path: name, Err: fs.ErrNotExist, Op: "stat"}
}
