package rpc

import (
	"a.yandex-team.ru/infra/hostctl/internal/units"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/storage"
	"a.yandex-team.ru/infra/hostctl/internal/yamlutil"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"encoding/json"
	"errors"
	"fmt"
	"github.com/golang/protobuf/jsonpb"
	"github.com/golang/protobuf/proto"
	"gopkg.in/yaml.v2"
	"strings"
)

func Apply(req *pb.ApplyRequest) (*pb.ApplyResponse, error) {
	l, err := CreateLog(req.Verbose, req.Logfile)
	if err != nil {
		return nil, err
	}
	marshaledReq, err := yamlutil.ProtoToYaml(req)
	if err != nil {
		return nil, err
	}
	sb := strings.Builder{}
	sb.WriteString("\n================ Started apply request =================\n")
	sb.WriteString("==================== Request params ====================\n")
	sb.Write(marshaledReq)
	sb.WriteString("\n========================================================\n")
	l.Infof(sb.String())
	var unitEnv *env.Env
	if req.JobEnv == pb.EnvType_NOOP {
		unitEnv = env.Noop(l)
	} else {
		unitEnv = env.Real(l, true, "", "")
	}
	if req.NoExecute {
		plans, err := units.Plan(unitEnv, req.Info, req.UnitFile, storage.NewReadonly(""))
		if err != nil {
			return nil, fmt.Errorf("failed to get plan: %w", err)
		}
		planYAML, err := yaml.Marshal(plans)
		if err != nil {
			return nil, errors.New("failed to yaml marshal plan")
		}
		planFmt := "\nExecution plan:\n"
		m := "--------------------"
		planFmt += m + "\n"
		planFmt += string(planYAML)
		planFmt += strings.Repeat("-", len(m))
		l.Infof(planFmt)
		return nil, nil
	}
	status, err := units.Apply(unitEnv, req.Info, req.UnitFile)
	if err != nil {
		return nil, fmt.Errorf("execution failed: %w", err)
	}
	statusYAML, err2 := yamlifyPB(status.Proto())
	if err2 != nil {
		return nil, fmt.Errorf("failed to yaml marshal status: %w", err2)
	} else {
		l.Infof("status:\n%s", statusYAML)
	}
	return &pb.ApplyResponse{
		Status: status.Proto(),
	}, nil
}

func yamlifyPB(m proto.Message) (string, error) {
	mrshlr := jsonpb.Marshaler{
		EnumsAsInts:  false,
		EmitDefaults: false,
		Indent:       "  ",
		OrigName:     true,
		AnyResolver:  nil,
	}
	jsonMessage, err := mrshlr.MarshalToString(m)
	if err != nil {
		return "", err
	}
	var messageInterface interface{}
	err = json.Unmarshal([]byte(jsonMessage), &messageInterface)
	if err != nil {
		return "", err
	}
	yamlMessage, err := yaml.Marshal(messageInterface)
	if err != nil {
		return "", err
	}
	return string(yamlMessage), nil
}
