package rpc

import (
	"time"

	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"

	"a.yandex-team.ru/library/go/core/log"
	azap "a.yandex-team.ru/library/go/core/log/zap"
)

const (
	/*
		Inherited from original ya-salt:
		 * need ms to be able to "profile"
		 * no need for TZ - we have MSK on all hosts
	*/
	timeLayout = "2006-01-02 15:04:05.000"
)

func timeFormat(t time.Time, enc zapcore.PrimitiveArrayEncoder) {
	// Code is taken from zapcore/encoder.go - found no other way.
	type appendTimeEncoder interface {
		AppendTimeLayout(time.Time, string)
	}

	if enc, ok := enc.(appendTimeEncoder); ok {
		enc.AppendTimeLayout(t, timeLayout)
		return
	}
	enc.AppendString(t.Format(timeLayout))
}

func newConfig(level log.Level) zap.Config {
	return zap.Config{
		Level:    zap.NewAtomicLevelAt(azap.ZapifyLevel(level)),
		Encoding: "console",
		EncoderConfig: zapcore.EncoderConfig{
			MessageKey:       "msg",
			LevelKey:         "level",
			TimeKey:          "ts",
			CallerKey:        "caller",
			NameKey:          "name",
			EncodeLevel:      zapcore.CapitalLevelEncoder,
			EncodeTime:       timeFormat,
			EncodeDuration:   zapcore.StringDurationEncoder,
			EncodeCaller:     zapcore.ShortCallerEncoder,
			ConsoleSeparator: " - ", // Same as in ya-salt
		},
		OutputPaths:      []string{"stdout"},
		ErrorOutputPaths: []string{"stderr"},
	}
}

func CreateLog(verbose bool, logfile string) (log.Logger, error) {
	logLevel := log.InfoLevel
	if verbose {
		logLevel = log.DebugLevel
	}
	cfg := newConfig(logLevel)
	if len(logfile) > 0 {
		// Use file path provided - use it
		cfg.OutputPaths = append(cfg.OutputPaths, logfile)
		cfg.ErrorOutputPaths = append(cfg.ErrorOutputPaths, logfile)
	}
	return azap.New(cfg)
}
