import contextlib
import logging
import os
import subprocess
import tempfile

import yaml

import sandbox.sdk2 as sandbox_sdk2
from ci.tasklet.common.proto import service_pb2 as ci
from infra.infractl.ci_tasklets.apply_specs.proto import apply_specs_tasklet
from sandbox.projects.common.vcs import arc as lib_arc
from tasklet.services.yav.proto import yav_pb2 as yav


log = logging.getLogger(__name__)


def run_command(cmd, env):
    environ = os.environ.copy()
    environ.update(env)
    if isinstance(cmd, list):
        cmd = subprocess.list2cmdline(cmd)

    log.info("run command `%s`", cmd)
    output = subprocess.check_output(cmd,
                                     stderr=subprocess.STDOUT,
                                     env=environ,
                                     close_fds=True,
                                     shell=True)
    log.info("`%s` output is:\n%s", cmd, output)


def cast_attributes_to_docker_build_arg(resource):
    attrs = resource.attributes
    package_path = attrs["package_path"]
    digest = attrs["docker_image_digest"]
    image = attrs["resource_version"]
    return ["-d", "{}={}@{}".format(package_path, image, digest)]


def cast_attributes_to_resource_build_arg(resource):
    attrs = resource.attributes
    package_path = attrs["package_path"]
    return ["-s", "{}={}".format(package_path, resource.id)]


def parse_iyaml_output(iyaml_path):
    with open(iyaml_path) as f:
        try:
            d = yaml.safe_load(f)
        except yaml.YAMLError:
            log.exception("failed to parse i.yaml file %s", iyaml_path)
            return ""
        return d.get("output")


class ApplyInfraCtlSpecsImpl(apply_specs_tasklet.ApplyInfraCtlSpecsBase):

    ARCADIA_PATH = "mounted_arcadia"

    INFRACTL_SPEC_RESOURCE_TYPE = "INFRACTL_SPEC"

    KUBECONFIG_TPL = """
apiVersion: v1
clusters:
- cluster:
    server: https://k.yandex-team.ru
  name: infractl
contexts:
- context:
    cluster: infractl
    user: user
  name: infractl
current-context: infractl
kind: Config
users:
- name: user
  user:
    token: {}
"""

    def _get_secret(self, key):
        secret_uid = self.input.context.secret_uid
        spec = yav.YavSecretSpec(uuid=secret_uid, key=key)
        try:
            return self.ctx.yav.get_secret(spec, default_key=key).secret
        except Exception as e:
            msg = str(e)
            # NOTE: seems it's the only way to determine that the reason of exception
            # is that the key was not found in YAV secret
            if key in msg:
                raise ValueError(
                    'Key "{key}" is not found in your CI-secret in YAV. '
                    'Please add "{key}" key into CI-secret '
                    'specified in your a.yaml file. For more instructions see: '
                    'https://docs.yandex-team.ru/infractl/howto#ci-prerequisites'.format(key=key)
                )
            raise

    def _download_resource(self, **constraints):
        resources = sandbox_sdk2.Resource.find(**constraints)
        assert resources, "Unable to find resources by {}".format(constraints)
        first = resources.first()
        assert first, "Unable to find first resource by {}".format(constraints)
        return str(sandbox_sdk2.ResourceData(first).path)

    def _make_kubeconfig(self):
        kube_token = self._get_secret("infractl_ci.kubetoken")
        return self.KUBECONFIG_TPL.format(kube_token)

    @contextlib.contextmanager
    def mount_arc(self):
        # https://st.yandex-team.ru/ARC-3966
        os.environ['ARC_SKIP_SERVER_CACHE'] = 'true'

        arc_token = self._get_secret("ci.token")
        arc = lib_arc.Arc(arc_oauth_token=arc_token)
        with arc.mount_path("", "trunk", mount_point=self.ARCADIA_PATH, fetch_all=False):
            arc_binary_path = arc.binary_path
            arcadia_path = os.path.abspath(self.ARCADIA_PATH)
            path = os.environ["PATH"] + os.pathsep + os.path.dirname(arc_binary_path) + os.pathsep + arcadia_path
            env = {
                'ARC_TOKEN': arc_token,
                'ARC_BIN': arc_binary_path,
                'ARCADIA_PATH': arcadia_path,
                'PATH': path
            }
            yield env

    def save_progress(self, progress_ratio, message, step_id, status):
        progress = ci.TaskletProgress()
        progress.job_instance_id.CopyFrom(self.input.context.job_instance_id)
        progress.id = step_id
        progress.progress = progress_ratio
        progress.text = message
        progress.status = status
        self.ctx.ci.UpdateProgress(progress)

    def docker_login(self):
        r = self.input.config.build.docker.registry
        u = self.input.config.build.docker.user
        t = self._get_secret("infractl_ci.docker_token")
        env = {"DOCKER_TOKEN": t}
        cmd = ["/usr/bin/docker", "login", r, "-u", u, "-p", "$DOCKER_TOKEN"]
        run_command(cmd, env)

    def infractl_pull(self, namespace, resources):
        cmd = [self._infractl_bin, "pull", "--kubeconfig", self._kubeconfig, namespace]
        assert resources, "Resources to pull list must not be empty. Please, set source.k8s.resources"
        for obj in resources:
            cmd.append(f"{obj.kind}/{obj.name}")
        run_command(cmd, env={})

    def infractl_make(self, iyaml_dir):
        iyaml_path = os.path.join(iyaml_dir, "i.yaml")
        if not os.path.exists(iyaml_path):
            raise Exception(f"Manifest file {iyaml_path} does not exist")

        output = parse_iyaml_output(iyaml_path)
        if not output:
            raise Exception(f"Manifest file {iyaml_path} does not contain output field")

        cmd = [self._infractl_bin, "make", "--kubeconfig", self._kubeconfig, "--arc-path", self._arc_env["ARCADIA_PATH"], iyaml_dir]
        run_command(cmd, env={})
        return os.path.join(iyaml_dir, output)

    def pull(self):
        self._pulled_specs = []
        source = self.input.config.source
        source_kind = source.WhichOneof('source')
        if source_kind == 'arc':
            if source.arc.manifest and source.arc.path:
                raise AssertionError("manifest and path fields cannot be set simultaneously. "
                                     "Please set only manifest field (path is deprecated).")
            if source.arc.manifest:
                iyaml_dir = os.path.join(self._arc_env["ARCADIA_PATH"], source.arc.manifest)
                self._pulled_specs.append(self.infractl_make(iyaml_dir))
            else:
                path = os.path.join(self._arc_env["ARCADIA_PATH"], source.arc.path)
                self._pulled_specs.append(path)
        elif source_kind == 'k8s' and source.k8s.namespace:
            self.infractl_pull(source.k8s.namespace, source.k8s.resources)
            self._pulled_specs.append("infra.*.yaml")
        else:
            raise AssertionError("Specs source field cannot be empty. Please set source.arc or source.k8s")

    def build(self):
        has_docker = False
        packages_args = []
        for r in self.input.sandbox_resources:
            if self.is_docker(r):
                has_docker = True
                packages_args.extend(cast_attributes_to_docker_build_arg(r))
            elif self.is_package(r):
                packages_args.extend(cast_attributes_to_resource_build_arg(r))
        if has_docker:
            self.docker_login()
        cmd = (
            [self._infractl_bin, "build", "--incremental", "--arc-path", self._arc_env["ARCADIA_PATH"]]
            + packages_args
            + self._pulled_specs
        )
        run_command(cmd, env={})

    def put(self):
        cmd = [self._infractl_bin, "put", "--kubeconfig", self._kubeconfig] + self._pulled_specs

        if self.input.config.put.message:
            cmd.extend(("--message", self.input.config.put.message))
        if self.input.config.put.wait.enabled:
            cmd.append("--wait")
            if self.input.config.put.wait.timeout != 0:
                cmd.extend(("--timeout", str(self.input.config.put.wait.timeout)))
        run_command(cmd, env={})

    def prepare_steps(self):
        self.steps = [
            (self.pull, "Pulling specs from Sandbox/K8S"),
            (self.build, "Running `infractl build`"),
            (self.put, "Running `infractl put`"),
        ]

    def prepare_resources(self, kubeconfig_f):
        self._infractl_bin = self._download_resource(id=self.input.config.infractl_bin_resource)
        log.info(
            "Downloaded infractl bin from resource %r to %r",
            self.input.config.infractl_bin_resource,
            self._infractl_bin,
        )
        kubeconfig_f.write(self._make_kubeconfig().encode("utf-8"))
        kubeconfig_f.flush()
        self._kubeconfig = kubeconfig_f.name

    def is_docker(self, resource):
        return resource.attributes.get('docker_image_digest') is not None

    def is_package(self, resource):
        return resource.attributes.get('package_path') is not None

    def run(self):
        with tempfile.TemporaryDirectory() as tmpd:
            kubeconfig_f = tempfile.NamedTemporaryFile(dir=tmpd)
            self.prepare_resources(kubeconfig_f=kubeconfig_f)
            self.prepare_steps()
            os.chdir(tmpd)
            with self.mount_arc() as arc_env:
                self._arc_env = arc_env
                err = None
                err_msg = None
                for idx, (step, message) in enumerate(self.steps):
                    log.info("[%2d] %s", idx, message)
                    self.save_progress(progress_ratio=idx / len(self.steps),
                                       message=message,
                                       step_id='progress',
                                       status=ci.TaskletProgress.Status.SUCCESSFUL)
                    try:
                        step()
                    except subprocess.CalledProcessError as e:
                        err = e
                        err_msg = "{}: {}".format(e, e.output)
                    except Exception as e:
                        err = e
                        err_msg = str(e)
                    if err is not None:
                        self.save_progress(
                            progress_ratio=1.,
                            message="ApplyInfraCtlSpecs failed: {}".format(err_msg),
                            step_id="progress",
                            status=ci.TaskletProgress.Status.FAILED,
                        )
                        self.output.state.success = False
                        self.output.state.message = "Step {} failed: {}".format(idx, err_msg)
                        raise err
        self.save_progress(
            progress_ratio=1.,
            message="Done.",
            step_id='progress',
            status=ci.TaskletProgress.Status.SUCCESSFUL,
        )
        self.output.state.success = True
        self.output.state.message = "Success"
