package bootstrap

import (
	"fmt"

	"github.com/spf13/cobra"
	corev1 "k8s.io/api/core/v1"
	crlog "sigs.k8s.io/controller-runtime/pkg/log"

	"a.yandex-team.ru/infra/infractl/cli/commands/bootstrap/internal"
	"a.yandex-team.ru/infra/infractl/cli/commands/root"
	yputil "a.yandex-team.ru/infra/infractl/cli/commands/util/yp"
	"a.yandex-team.ru/infra/infractl/internal/secrets"
)

var nullLog = crlog.Log

func Bootstrap() *cobra.Command {
	var ypCluster = yputil.XDC
	var username string

	cmd := &cobra.Command{
		Use:              "bootstrap",
		Short:            "Bootstrap new InfraCtl service",
		TraverseChildren: true,
		Args:             cobra.NoArgs,
		Run: func(cmd *cobra.Command, args []string) {
			// First print greetings, then print error message if we couldn't
			// determine current user and only after it do the rest
			fmt.Println("Welcome to infractl!")
			env := internal.MustDetectEnv(username)
			fmt.Println("This wizard will help you create simple namespace: starting point for your project.")
			clients := internal.NewClients(ypCluster.String())

			objects := runNamespaceAndProject(ypCluster, env.User)
			for _, obj := range objects {
				mustPutObject(clients.Kube(), obj)
			}
			ns := objects[0].(*corev1.Namespace)

			creds := &secrets.Credentials{}
			internal.AskAccessToProvider(root.Context, clients, creds, "yp", ns)

			runtimes := runRuntime(ns, creds, ypCluster)

			msg := makePutObjectsMessage(runtimes...)
			if ans := internal.Confirm(msg); ans {
				for _, o := range runtimes {
					mustPutObject(clients.Kube(), o)
				}
			}

			fmt.Println("Bootstrap finished!")
		},
	}
	cmd.AddCommand(bootstrapNamespace())
	cmd.AddCommand(bootstrapRuntime())
	cmd.Flags().Var(&ypCluster, "cluster", "YP cluster for searching objects in")
	cmd.Flags().StringVarP(&username, "user", "u", "", "username to execute bootstrap and create objects from")
	return cmd
}
