package bootstrap

import (
	"fmt"
	"strings"

	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/cli/commands/bootstrap/internal"
	"a.yandex-team.ru/infra/infractl/cli/commands/bootstrap/internal/objects"
	"a.yandex-team.ru/infra/infractl/cli/commands/root"
)

type Bootstrapper struct {
	clients *internal.Clients
	maker   ObjectsMaker
}

func (b *Bootstrapper) Bootstrap() ([]client.Object, error) {
	baseName := ""
	isRetry := false
	var err error
	for baseName == "" {
		if baseName, err = b.makeUniqueBaseName(isRetry); err != nil {
			return nil, err
		}
		isRetry = true
	}
	objs, err := b.maker.Make(baseName)
	if err != nil {
		return nil, fmt.Errorf("failed to generate object specs: %w", err)
	}
	for _, obj := range objs {
		if err = b.maker.Validate(obj); err != nil {
			return nil, err
		}
	}
	return objs, nil
}

func makeReadableID(id objects.Ref) string {
	return fmt.Sprintf("%s %q", id.GVK.Kind, id.Name.Name)
}

func makeOverwritePrompt(ids []objects.Ref) string {
	if len(ids) == 1 {
		return fmt.Sprintf("%s already exists. Overwrite it?", makeReadableID(ids[0]))
	}
	quoted := make([]string, 0, len(ids))
	for _, id := range ids {
		quoted = append(quoted, makeReadableID(id))
	}
	s := strings.Join(quoted, " and ")
	return fmt.Sprintf("%s already exist. Overwrite them?", s)
}

func (b *Bootstrapper) makeUniqueBaseName(isRetry bool) (string, error) {
	baseName, ids := b.maker.MakeObjectRefs(isRetry)
	found, err := listExistingObjectNames(root.Context, b.clients.Kube(), ids)
	if err != nil {
		return "", err
	}
	if len(found) == 0 {
		return baseName, nil
	}
	msg := makeOverwritePrompt(found)
	if internal.Confirm(msg) {
		return baseName, nil
	}
	return "", nil
}
