package internal

import (
	"fmt"
	"log"

	"github.com/AlecAivazis/survey/v2"
)

var askOpts = []survey.AskOpt{
	survey.WithIcons(func(icons *survey.IconSet) {
		icons.Question.Text = "→"
	}),
}

func makeAskOpts(required bool, validators ...survey.Validator) []survey.AskOpt {
	opts := make([]survey.AskOpt, 0, len(askOpts)+len(validators))
	if required {
		opts = append(opts, survey.WithValidator(survey.Required))
	}
	opts = append(opts, askOpts...)
	for _, validator := range validators {
		opts = append(opts, survey.WithValidator(validator))
	}
	return opts
}

func AskRequired(msg string, validators ...survey.Validator) string {
	return Ask(msg, true, validators...)
}

func Ask(msg string, required bool, validators ...survey.Validator) string {
	ans := ""
	prompt := &survey.Input{Message: msg}
	if err := survey.AskOne(prompt, &ans, makeAskOpts(required, validators...)...); err != nil {
		log.Fatalf(err.Error())
	}
	return ans
}

func AskDefault(msg, defaultValue string, helpMessage string, validators ...survey.Validator) string {
	if helpMessage == "" {
		msg = fmt.Sprintf("%s (default: %q)", msg, defaultValue)
	} else {
		msg = fmt.Sprintf("%s (default: %q, %s)", msg, defaultValue, helpMessage)
	}
	ans := Ask(msg, false, validators...)
	if ans == "" {
		return defaultValue
	}
	return ans
}

func AskSelect(msg string, opts ...string) string {
	ans := ""
	prompt := &survey.Select{
		Message: msg,
		Options: opts,
	}
	if err := survey.AskOne(prompt, &ans, askOpts...); err != nil {
		log.Fatalf(err.Error())
	}
	return ans
}

func Confirm(msg string) bool {
	ans := false
	prompt := &survey.Confirm{
		Message: msg,
	}
	if err := survey.AskOne(prompt, &ans); err != nil {
		log.Fatalf(err.Error())
	}
	return ans
}
