package internal

import (
	"context"
	"fmt"
	"log"

	corev1 "k8s.io/api/core/v1"

	"a.yandex-team.ru/infra/infractl/internal/secrets"
	proxyclient "a.yandex-team.ru/infra/infractl/services/oauth-to-login/client"
)

func getPasswordFromSecret(ctx context.Context, clients *Clients, secret string) (string, error) {
	rsp, err := clients.Yav().GetVersion(ctx, secret)
	if err != nil {
		return "", fmt.Errorf("failed to find robot secret in yav: %w", err)
	}
	for _, v := range rsp.Version.Values {
		if v.Key == "password" {
			return v.Value, nil
		}
	}
	return "", fmt.Errorf("no robot secrets with key \"password\" found")
}

func getPasswordFromSecrets(ctx context.Context, clients *Clients, secrets []string, opts secrets.FindSecretsOptions) (string, error) {
	if len(secrets) == 0 {
		return "", fmt.Errorf("no robot secret found by alias %q", opts.Alias)
	}
	if len(secrets) > 1 {
		return "", fmt.Errorf("found more than 1 robot secrets by alias %q", opts.Alias)
	}
	secret := secrets[0]
	return getPasswordFromSecret(ctx, clients, secret)
}

func getLoginBySecretUUID(ctx context.Context, clients *Clients, provider, secretUUID string) (string, error) {
	rsp, err := clients.Yav().GetVersion(ctx, secretUUID)
	if err != nil {
		return "", fmt.Errorf("failed to find yav secret for %q: %w", provider, err)
	}
	prov := secrets.GetProvider(provider)
	if prov == nil {
		return "", fmt.Errorf("secrets provider %q not found", provider)
	}

	var token string
	for _, v := range rsp.Version.Values {
		if v.Key == prov.YAVSecretKey {
			token = v.Value
			break
		}
	}
	if token == "" {
		return "", fmt.Errorf("yav secret %q does not contain key %q", secretUUID, prov.YAVSecretKey)
	}

	c := proxyclient.NewProductionClient()
	info, err := c.GetInfo(ctx, token)
	if err != nil {
		return "", fmt.Errorf("failed to get login by token in yav secret %s#%s", secretUUID, prov.YAVSecretKey)
	}
	return info.Login, nil
}

func askSecretForPassword(ctx context.Context, clients *Clients, robotName, askReason string) string {
	msg := fmt.Sprintf("InfraCtl cannot find robot's password secret: %s. Please enter secret UUID for %q manually:", askReason, robotName)
	secretUUID := AskRequired(msg)
	password, err := getPasswordFromSecret(ctx, clients, secretUUID)
	if err != nil {
		log.Fatal(err.Error())
	}
	return password
}

func askCredentials(ctx context.Context, clients *Clients, provider, robotName string) secrets.Credentials {
	opts := secrets.FindSecretsOptions{}
	if robotName == "" {
		msg := "Enter robot name on behalf of which InfraCtl will do requests to external systems (providers) like YP, CI, docker registry, etc.\n>"
		robotName = AskRequired(msg)
	}
	opts.Alias = robotName
	if opts.Author == "" {
		opts.Author = "robot-secrets"
	}
	secretUUIDs, err := clients.Yav().FindSecrets(ctx, opts)
	var askPasswordReason, password string
	if err == nil {
		password, err = getPasswordFromSecrets(ctx, clients, secretUUIDs, opts)
		if err != nil {
			askPasswordReason = err.Error()
		}
	} else {
		askPasswordReason = fmt.Sprintf("failed to find robot secrets in yav for %q: %v", robotName, err)
	}
	if askPasswordReason != "" {
		password = askSecretForPassword(ctx, clients, robotName, askPasswordReason)
	}
	return secrets.Credentials{Username: robotName, Password: password}
}

func getCurrentRobotWithAccess(ctx context.Context, clients *Clients, provider string, ns *corev1.Namespace) (string, string) {
	sec, err := secrets.GetKubeSecret(ctx, clients.Kube(), provider, ns)
	if err != nil {
		log.Fatalf("Failed to check if secret for %s-token exists in k8s: %v", provider, err)
	}
	if sec == nil {
		return "", ""
	}
	var secUUID string
	if secUUIDBytes, ok := sec.Data[secrets.SecretUUIDDataKey]; ok {
		secUUID = string(secUUIDBytes)
	}
	if secUUID == "" {
		return "", ""
	}
	// ignore error, just ask robotname again if failed to get it by secret UUID
	curRobotName, _ := getLoginBySecretUUID(ctx, clients, provider, secUUID)
	return curRobotName, secUUID
}

// AskAccessToProvider has side effect: it changes creds argument. So creds cannot be nil
func AskAccessToProvider(ctx context.Context, clients *Clients, creds *secrets.Credentials, provider string, ns *corev1.Namespace) string {
	// firstly check if robot already has access
	curRobotName, secUUID := getCurrentRobotWithAccess(ctx, clients, provider, ns)

	// we already have robot with access
	if curRobotName != "" {
		if creds.Username == "" {
			// save current robot with access to creds. This way we will not ask robotname further
			creds.Username = curRobotName
			return secUUID
		}
		// do not change robotname if creds already has it, just return current secret UUID
		if creds.Username == curRobotName {
			// robot already has access to provider
			return secUUID
		}
	}

	if creds.Username == "" || creds.Password == "" {
		// ask credentials from user
		*creds = askCredentials(ctx, clients, provider, creds.Username)
	}

	// give access by credentials
	secUUID, err := secrets.GiveAccessToProviderByCredentials(ctx, clients.Yav(), creds, secrets.EnsureSecretOptions{
		KubeClient:       clients.Kube(),
		YAVClient:        clients.Yav(),
		Provider:         provider,
		Namespace:        ns,
		TVMAliasesAndIDs: []string{"deployctl"},
	})
	if err != nil {
		log.Fatal(err.Error())
	}
	fmt.Printf("Successfully saved %s-token for %q to YAV: https://yav.yandex-team.ru/secret/%s/explore/versions\n", provider, creds.Username, secUUID)
	return secUUID
}
