package bootstrap

import (
	"fmt"
	"log"

	"github.com/spf13/cobra"
	corev1 "k8s.io/api/core/v1"
	"sigs.k8s.io/controller-runtime/pkg/client"
	crlog "sigs.k8s.io/controller-runtime/pkg/log"

	"a.yandex-team.ru/infra/infractl/cli/commands/bootstrap/internal"
	"a.yandex-team.ru/infra/infractl/cli/commands/bootstrap/namespace"
	"a.yandex-team.ru/infra/infractl/cli/commands/root"
	yputil "a.yandex-team.ru/infra/infractl/cli/commands/util/yp"
	"a.yandex-team.ru/infra/infractl/clients/abc"
	"a.yandex-team.ru/infra/infractl/internal/secrets"
	"a.yandex-team.ru/infra/infractl/internal/validation"
)

func bootstrapNamespace() *cobra.Command {
	var ypCluster = yputil.XDC
	var username string

	cmd := &cobra.Command{
		Use:   "namespace",
		Short: "Bootstrap namespace and Yandex.Deploy project",
		Args:  cobra.NoArgs,
		Run: func(cmd *cobra.Command, args []string) {
			// First print greetings, then print error message if we couldn't
			// determine current user and only after it do the rest
			fmt.Println("Welcome to infractl!")
			env := internal.MustDetectEnv(username)
			clients := internal.NewClients(string(ypCluster))

			objects := runNamespaceAndProject(ypCluster, env.User)
			for _, obj := range objects {
				mustPutObject(clients.Kube(), obj)
			}
			ns := objects[0].(*corev1.Namespace)

			creds := &secrets.Credentials{}
			internal.AskAccessToProvider(root.Context, clients, creds, "yp", ns)

			fmt.Println("Bootstrap namespace finished!")
		},
	}

	cmd.Flags().Var(&ypCluster, "cluster", "YP cluster for searching objects in")
	cmd.Flags().StringVarP(&username, "user", "u", "", "username to execute bootstrap and create objects from")
	return cmd
}

func askABCSlug(client *abc.Client, username string) string {
	validator := func(ans interface{}) error {
		slug, ok := ans.(string)
		if !ok {
			return fmt.Errorf("cannot cast input to string")
		}
		if _, err := client.GetServiceIDBySlug(slug); err != nil {
			return err
		}
		isValid, msg, err := validation.ValidateIsUserABCMember(client, crlog.Log, slug, username)
		if err != nil {
			return err
		}
		if !isValid {
			return fmt.Errorf(msg)
		}
		return nil
	}
	return internal.AskRequired("Enter ABC service slug to attach your namespace:", validator)
}

func runNamespaceAndProject(ypCluster yputil.YpCluster, username string) []client.Object {
	clients := internal.NewClients(ypCluster.String())

	// First validate if user is allowed to use this ABC to fail fast
	// and even not asking namespace name
	abcSlug := askABCSlug(clients.Abc(), username)
	maker := namespace.NewMaker(abcSlug, clients, ypCluster)
	fmt.Println("We will create new directory to store your specs, infractl namespace and Yandex.Deploy project with the same name.")

	nsBootstrapper := &Bootstrapper{
		clients: clients,
		maker:   maker,
	}

	objects, err := nsBootstrapper.Bootstrap()
	if err != nil {
		log.Fatalf("Bootstrap failed: %v", err)
	}
	return objects
}
