package namespace

import (
	"errors"
	"fmt"
	"log"
	"os"

	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/runtime/schema"
	"k8s.io/apimachinery/pkg/types"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/cli/commands/bootstrap/internal"
	"a.yandex-team.ru/infra/infractl/cli/commands/bootstrap/internal/objects"
	yputil "a.yandex-team.ru/infra/infractl/cli/commands/util/yp"
	"a.yandex-team.ru/infra/infractl/controllers/deploy/api/project/proto_v1"
	pv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/project/v1"
	"a.yandex-team.ru/infra/infractl/internal/labels"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

type maker struct {
	clients           *internal.Clients
	abcSlug           string
	monitoringProject string
	nsValidator       Validator
	prjValidator      ProjectValidator
}

func NewMaker(abcSlug string, clients *internal.Clients, ypCluster yputil.YpCluster) *maker {
	// TODO(reddi): ask monitoring project from user as soon as it will be required
	// TODO(reddi): (for now it is required only for test and prestable).
	monitoringProject := ""
	if ypCluster != yputil.XDC {
		monitoringProject = "sample-bootstrap-project"
	}
	return &maker{
		clients:           clients,
		abcSlug:           abcSlug,
		monitoringProject: monitoringProject,
		nsValidator:       Validator{Clients: clients},
		prjValidator:      ProjectValidator{Clients: clients, abcSlug: abcSlug},
	}
}

func isSuitableDir(name string) bool {
	if _, err := os.Stat(name); errors.Is(err, os.ErrNotExist) {
		return true
	}
	if !internal.Confirm(fmt.Sprintf("Directory %q already exists. Would you like to remove it?", name)) {
		return false
	}
	if err := os.RemoveAll(name); err != nil {
		log.Fatalf("cannot remove %q directory: %v", name, err)
	}
	return true
}

func (m *maker) MakeObjectRefs(isRetry bool) (string, []objects.Ref) {
	var name string
	helpMessage := ""
	for {
		if isRetry {
			helpMessage = "Ctrl+C to exit"
		}
		name = internal.AskDefault("Enter namespace name", m.abcSlug, helpMessage)
		// Validate that directory with the same name does not exist
		if isSuitableDir(name) {
			break
		}
		isRetry = true
	}
	return name, []objects.Ref{
		{
			GVK: schema.GroupVersionKind{
				Group:   corev1.SchemeGroupVersion.Group,
				Version: corev1.SchemeGroupVersion.Version,
				Kind:    "Namespace",
			},
			Name: types.NamespacedName{
				Name:      name,
				Namespace: "",
			},
		},
		{
			GVK: schema.GroupVersionKind{
				Group:   pv1.GroupVersion.Group,
				Version: pv1.GroupVersion.Version,
				Kind:    "DeployProject",
			},
			Name: types.NamespacedName{
				Name:      name,
				Namespace: name,
			},
		},
	}
}

func (m *maker) Make(baseName string) ([]client.Object, error) {
	ns := &corev1.Namespace{}
	ns.SetGroupVersionKind(schema.GroupVersionKind{
		Group:   corev1.SchemeGroupVersion.Group,
		Version: corev1.SchemeGroupVersion.Version,
		Kind:    "Namespace",
	})
	ns.Name = baseName
	ns.SetLabels(map[string]string{labels.ABC: m.abcSlug})

	p := &pv1.DeployProject{
		Spec: &proto_v1.Spec{
			ProjectSpec: &ypapi.TProjectSpec{MonitoringProject: m.monitoringProject},
		},
	}
	p.Namespace = baseName
	p.Name = baseName
	p.SetGroupVersionKind(schema.GroupVersionKind{
		Group:   pv1.GroupVersion.Group,
		Version: pv1.GroupVersion.Version,
		Kind:    "DeployProject",
	})
	return []client.Object{ns, p}, nil
}

func (m *maker) Validate(object client.Object) error {
	switch object.(type) {
	case *corev1.Namespace:
		return m.nsValidator.Validate(object)
	case *pv1.DeployProject:
		return m.prjValidator.Validate(object)
	}
	return fmt.Errorf("unexpected object type: %T", object)
}
