package bootstrap

import (
	"context"
	"fmt"
	"log"
	"strings"

	"k8s.io/apimachinery/pkg/apis/meta/v1/unstructured"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/cli/commands/bootstrap/internal/objects"
	"a.yandex-team.ru/infra/infractl/cli/commands/root"
	"a.yandex-team.ru/infra/infractl/internal/labels"
	"a.yandex-team.ru/infra/infractl/util/kubeutil"
)

func listExistingObjectNames(
	ctx context.Context,
	kubeclient *kubeutil.Client,
	ids []objects.Ref,
) ([]objects.Ref, error) {
	var found []objects.Ref
	for _, id := range ids {
		obj := &unstructured.Unstructured{}
		obj.SetGroupVersionKind(id.GVK)
		if err := kubeclient.Get(ctx, id.Name, obj); err != nil {
			if client.IgnoreNotFound(err) == nil {
				continue
			}
			return nil, err
		}
		found = append(found, id)
	}
	return found, nil
}

func mustPutObject(kubeClient *kubeutil.Client, o client.Object) {
	kind := strings.ToLower(o.GetObjectKind().GroupVersionKind().Kind)
	lbls := o.GetLabels()
	if lbls == nil {
		lbls = map[string]string{}
	}
	if _, exists := lbls[labels.SetupByLabel]; !exists {
		lbls[labels.SetupByLabel] = "INFRACTL_BOOTSTRAP"
		o.SetLabels(lbls)
	}
	fmt.Printf("Put %s %q in K8S...\n", kind, o.GetName())
	if _, err := kubeClient.PutObject(root.Context, client.ObjectKeyFromObject(o), o); err != nil {
		log.Fatalf("Failed to put %s %q in K8S: %v", kind, o.GetName(), err)
	}
	fmt.Println("Done.")
}

func makePutObjectsMessage(objects ...client.Object) string {
	parts := make([]string, 0, len(objects)+2)
	parts = append(parts, "Objects to put in K8S:\n")
	for _, o := range objects {
		parts = append(parts, fmt.Sprintf("%s %q", o.GetObjectKind().GroupVersionKind().Kind, o.GetName()))
	}
	parts = append(parts, "\nPut them all to K8S?")
	return strings.Join(parts, "\n")
}
