package bootstrap

import (
	"fmt"
	"log"

	"github.com/spf13/cobra"
	corev1 "k8s.io/api/core/v1"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/cli/commands/bootstrap/internal"
	"a.yandex-team.ru/infra/infractl/cli/commands/bootstrap/namespace"
	"a.yandex-team.ru/infra/infractl/cli/commands/bootstrap/runtime"
	"a.yandex-team.ru/infra/infractl/cli/commands/root"
	yputil "a.yandex-team.ru/infra/infractl/cli/commands/util/yp"
	"a.yandex-team.ru/infra/infractl/internal/labels"
	"a.yandex-team.ru/infra/infractl/internal/secrets"
)

func bootstrapRuntime() *cobra.Command {
	var ypCluster = yputil.XDC
	var nsName string

	cmd := &cobra.Command{
		Use:   "runtime",
		Short: "Bootstrap runtime",
		Args:  cobra.NoArgs,
		Run: func(cmd *cobra.Command, args []string) {
			clients := internal.NewClients(string(ypCluster))

			if nsName == "" {
				nsName = internal.AskRequired("Enter namespace name:")
			}
			ns := internal.MustGetNamespace(clients.Kube(), nsName)

			creds := &secrets.Credentials{}
			runtimes := runRuntime(ns, creds, ypCluster)

			internal.AskAccessToProvider(root.Context, clients, creds, "yp", ns)

			msg := makePutObjectsMessage(runtimes...)
			if ans := internal.Confirm(msg); ans {
				for _, o := range runtimes {
					mustPutObject(clients.Kube(), o)
				}
			}

			fmt.Println("Bootstrap runtime finished!")
		},
	}
	cmd.Flags().StringVarP(&nsName, "namespace", "n", "", "Namespace in which to create runtime")
	cmd.Flags().Var(&ypCluster, "cluster", "YP cluster")
	return cmd
}

func runRuntime(ns *corev1.Namespace, creds *secrets.Credentials, ypCluster yputil.YpCluster) []client.Object {

	clients := internal.NewClients(string(ypCluster))

	nsName := ns.GetName()
	nsValidator := &namespace.Validator{Clients: clients}
	// Check that user is creating runtime in ABC service he belongs to
	if err := nsValidator.Validate(ns); err != nil {
		log.Fatalf(err.Error())
	}

	rtMaker := runtime.NewMaker(
		clients,
		nsName,
		ns.Labels[labels.ABC],
		creds,
		ypCluster.String(),
	)

	rtBootstrapper := &Bootstrapper{
		clients: clients,
		maker:   rtMaker,
	}
	objects, err := rtBootstrapper.Bootstrap()
	if err != nil {
		log.Fatalf("Bootstrap failed: %v", err)
	}
	return objects
}
