package runtime

import (
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"

	"k8s.io/apimachinery/pkg/runtime/schema"
	"k8s.io/apimachinery/pkg/types"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/cli/commands/bootstrap/internal"
	"a.yandex-team.ru/infra/infractl/cli/commands/bootstrap/internal/objects"
	"a.yandex-team.ru/infra/infractl/cli/commands/root"
	"a.yandex-team.ru/infra/infractl/cli/commands/util/yp"
	"a.yandex-team.ru/infra/infractl/cli/internal/arcutil"
	"a.yandex-team.ru/infra/infractl/cli/internal/ayaml"
	"a.yandex-team.ru/infra/infractl/cli/internal/iyaml"
	substitutio "a.yandex-team.ru/infra/infractl/cli/internal/substitutions"
	"a.yandex-team.ru/infra/infractl/controllers/runtime/api/proto_v1"
	v1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	"a.yandex-team.ru/infra/infractl/internal/secrets"
	"a.yandex-team.ru/yp/go/proto/podagent"
)

const (
	dockerBuildPath       = "infra/deploy_ci/examples/docker_pipeline/build.json"
	dockerImageRepository = "docker-ci"
	appDockerImage        = "registry.yandex.net/deploy-ci/example_http_server:30"

	layersBuildPath = "infra/deploy_ci/examples/layer_pipeline_without_approves/build.json"
	baseLayer       = "2061235470"
	appLayer        = "3218997353"

	exampleConfigTemplate = `run:
    dev_mode: %v`
)

type maker struct {
	clients   *internal.Clients
	nsName    string
	abcSlug   string
	creds     *secrets.Credentials
	ypCluster string
}

func NewMaker(
	clients *internal.Clients,
	nsName string,
	abcSlug string,
	creds *secrets.Credentials,
	ypCluster string,
) *maker {
	return &maker{
		clients:   clients,
		nsName:    nsName,
		abcSlug:   abcSlug,
		creds:     creds,
		ypCluster: ypCluster,
	}
}

func (m *maker) makeEmptyRuntime(baseName string) *v1.Runtime {
	rt := &v1.Runtime{}
	rt.Name = baseName
	rt.Namespace = m.nsName
	rt.SetGroupVersionKind(schema.GroupVersionKind{
		Group:   v1.GroupVersion.Group,
		Version: v1.GroupVersion.Version,
		Kind:    "Runtime",
	})
	return rt
}

func (m *maker) makeBaseRuntimeFromInput(baseName string) *rtResult {
	rt := &v1.Runtime{}
	rt.SetGroupVersionKind(schema.GroupVersionKind{
		Group:   v1.GroupVersion.Group,
		Version: v1.GroupVersion.Version,
		Kind:    "Runtime",
	})
	rt.Name = baseName
	rt.Namespace = m.nsName

	rt.Spec = &proto_v1.Spec{}
	rt.Spec.NetworkId = internal.AskRequired("Enter network macro:")

	dockerOpt := "docker"
	layersOpt := "layers"
	image := internal.AskSelect("Choose image type:", dockerOpt, layersOpt)
	switch image {
	case dockerOpt:
		rt.Spec.Image = fmt.Sprintf("${package:%s}", dockerBuildPath)
	case layersOpt:
		rt.Spec.Layers = []*proto_v1.Spec_PortoLayer{
			{
				Id:         "base",
				ResourceId: baseLayer,
			},
			{
				Id:         "http_server",
				ResourceId: fmt.Sprintf("${resource:%s}", layersBuildPath),
			},
		}
		rt.Spec.Command = "/http_server"
	}

	rt.Spec.Readiness = &proto_v1.Spec_Probe{Probe: &proto_v1.Spec_Probe_Http{Http: &podagent.THttpGet{Path: "/readyz", Port: 8080, Answer: &podagent.THttpGet_Any{Any: true}}}}

	rt.Spec.Liveness = &proto_v1.Spec_Probe{Probe: &proto_v1.Spec_Probe_Command{Command: &proto_v1.Spec_Command{Cmd: "/bin/true"}}}

	rt.Spec.Compute = &proto_v1.Spec_ComputeResources{
		Vcpu:   "100",
		Memory: "500M",
		Net:    "1M",
	}

	rt.Spec.Storage = &proto_v1.Spec_Storage{
		StorageClass: "hdd",
		Quota:        "1G",
		IoBandwidth:  "1M",
	}

	var cluster string
	if m.ypCluster == string(yp.XDC) {
		cluster = "sas"
	} else {
		cluster = m.ypCluster
	}
	rt.Spec.Replicas = map[string]uint32{cluster: 1}

	return &rtResult{rt: rt, imageType: image}
}

func (m *maker) MakeObjectRefs(_ bool) (string, []objects.Ref) {
	baseName := internal.AskRequired("Enter base runtime name (two runtimes will be generated: with test- and prod- prefixes):")
	gvk := schema.GroupVersionKind{
		Group:   v1.GroupVersion.Group,
		Version: v1.GroupVersion.Version,
		Kind:    "Runtime",
	}
	var refs []objects.Ref
	names := []string{fmt.Sprintf("test-%s", baseName), fmt.Sprintf("prod-%s", baseName)}
	for _, name := range names {
		name := types.NamespacedName{Name: name, Namespace: m.nsName}
		refs = append(refs, objects.Ref{GVK: gvk, Name: name})
	}
	return baseName, refs
}

func (m *maker) generateAYAML(arcRoot string, rt *v1.Runtime, g ayaml.Generator, paths *specPaths) ([]byte, error) {
	prestablePath, err := filepath.Rel(arcRoot, paths.prestable)
	if err != nil {
		return nil, err
	}
	stablePath, err := filepath.Rel(arcRoot, paths.stable)
	if err != nil {
		return nil, err
	}
	group := internal.AskRequired("Enter sandbox group:")

	ns := internal.MustGetNamespace(m.clients.Kube(), m.nsName)

	secUUID := internal.AskAccessToProvider(root.Context, m.clients, m.creds, "ci", ns)
	_ = internal.AskAccessToProvider(root.Context, m.clients, m.creds, "infractl-ci", ns)

	opts := ayaml.GenerateOptions{
		SandboxGroup:      group,
		SecretUUID:        secUUID,
		SpecPathPrestable: prestablePath,
		SpecPathStable:    stablePath,
	}
	switch g := g.(type) {
	case *ayaml.SandboxGenerator:
		return g.Generate(rt, opts)
	case *ayaml.DockerGenerator:
		dockerSecUUID := internal.AskAccessToProvider(root.Context, m.clients, m.creds, "docker-registry", ns)
		g.User = m.creds.Username
		g.DockerYAVToken = fmt.Sprintf("%s#infractl_ci.docker_token", dockerSecUUID)
		return g.Generate(rt, opts)
	}
	return nil, fmt.Errorf("invalid type of generated runtime (neither sandbox layer nor docker)")
}

func (m *maker) Make(baseName string) ([]client.Object, error) {
	arcRoot, err := arcutil.ArcadiaRoot(".")
	if err != nil {
		return nil, fmt.Errorf("arcadia root is not found: %w", err)
	}

	// First let's create base runtime spec
	substs := substitutio.NewSubstitutions()

	var ayamlGen ayaml.Generator
	bg := ayaml.BaseGenerator{
		ABCSlug: m.abcSlug,
	}

	rtRes := m.makeBaseRuntimeFromInput(baseName)
	switch rtRes.imageType {
	case "docker":
		substs.DockerPackages[dockerBuildPath] = appDockerImage

		bg.BuildFilePath = dockerBuildPath
		ayamlGen = &ayaml.DockerGenerator{
			BaseGenerator:   bg,
			ImageRepository: dockerImageRepository,
		}
	case "layers":
		substs.SandboxResources[layersBuildPath] = &substitutio.SandboxResource{ResourceID: appLayer}

		bg.BuildFilePath = layersBuildPath
		ayamlGen = &ayaml.SandboxGenerator{BaseGenerator: bg}
	}

	// Here we generate all the specs of our service and dump them onto disk:
	// * i.yaml for base layer, test and prod
	// * runtime object base spec and patches for test and prod
	// * example of config file to be mounted into containers mountns
	// * .build.yaml containing prebuilt default docker image and layer
	paths, err := m.dumpSpecs(arcRoot, rtRes.rt, substs)
	if err != nil {
		return nil, err
	}

	// Now we run "ya tool infractl make" for test and prod and collect resulting compiled runtime specs
	var result []client.Object
	proc := iyaml.NewManifestProcessor(arcRoot, *m.clients.Kube())
	for _, dir := range []string{paths.prestable, paths.stable} {
		br, err := proc.Build(dir)
		if err != nil {
			return nil, fmt.Errorf("failed to process runtime manifest for %s: %w", dir, err)
		}
		for _, obj := range br.GetObjects() {
			if err = substitutio.AugmentSpec(obj, substs); err != nil {
				return nil, fmt.Errorf("failed to build %q runtime spec: %w", obj.GetName(), err)
			}
			result = append(result, obj)
		}
	}

	if len(result) == 0 {
		return result, nil
	}

	// Finally: generate a.yaml
	ans := internal.Confirm("Generate a.yaml?")
	if !ans {
		return result, nil
	}

	rt := result[0].(*v1.Runtime)
	cwd, err := os.Getwd()
	if err != nil {
		return nil, fmt.Errorf("failed to get cwd")
	}

	ayamlBytes, err := m.generateAYAML(arcRoot, rt, ayamlGen, paths)
	if err != nil {
		return nil, fmt.Errorf("failed to generate a.yaml: %w", err)
	}

	d := internal.AskDefault("Enter directory to save a.yaml (relative to cwd):", cwd, "")
	p, err := filepath.Abs(d)
	if err != nil {
		return nil, fmt.Errorf("failed to get abs path for directory %q: %w", d, err)
	}

	if err = ioutil.WriteFile(filepath.Join(p, "a.yaml"), ayamlBytes, os.ModePerm); err != nil {
		return nil, fmt.Errorf("failed to save a.yaml: %w", err)
	}

	return result, nil
}

func (m *maker) Validate(_ client.Object) error {
	return nil
}

type rtResult struct {
	rt        *v1.Runtime
	imageType string
}

type specPaths struct {
	prestable string
	stable    string
}
