package runtime

import (
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"path"
	"path/filepath"
	"strings"

	"github.com/ghodss/yaml"
	"google.golang.org/protobuf/encoding/protojson"
	"google.golang.org/protobuf/types/known/structpb"

	"a.yandex-team.ru/infra/infractl/cli/commands/make/manifest"
	substitutio "a.yandex-team.ru/infra/infractl/cli/internal/substitutions"
	"a.yandex-team.ru/infra/infractl/controllers/runtime/api/proto_v1"
	v1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
)

func dumpIYAML(m *manifest.Manifest) ([]byte, error) {
	js, err := protojson.Marshal(m)
	if err != nil {
		return nil, err
	}
	manifestDump, err := yaml.JSONToYAML(js)
	if err != nil {
		return nil, err
	}
	return manifestDump, nil
}

func (m *maker) makeBaseIYAML(runtimeFilename string) ([]byte, error) {
	result := &manifest.Manifest{
		Resources: []*manifest.Resource{
			{Path: &manifest.Resource_File{File: runtimeFilename}},
		},
	}
	return dumpIYAML(result)
}

func (m *maker) makePatchIYAML(basePath string, runtimeFilename string, env string) ([]byte, error) {
	ks, err := structpb.NewStruct(map[string]interface{}{
		"namePrefix":            fmt.Sprintf("%v-", env),
		"patchesStrategicMerge": []interface{}{runtimeFilename},
	})
	if err != nil {
		return nil, fmt.Errorf("cannot create patch description: %w", err)
	}

	result := &manifest.Manifest{
		Resources: []*manifest.Resource{
			{Path: &manifest.Resource_Base{Base: basePath}},
		},
		Patches: []*manifest.Patch{
			{Kustomize: ks},
			{Files: map[string]string{"/configs/cfg.yml": "cfg.yml"}},
		},
	}
	return dumpIYAML(result)
}

func (m *maker) makeRuntimePatch(base *v1.Runtime, env map[string]string, multiplier uint32) ([]byte, error) {
	patch := m.makeEmptyRuntime(base.GetName())
	patch.Spec = &proto_v1.Spec{
		Replicas: make(map[string]uint32, len(base.Spec.Replicas)),
		Env:      env,
	}
	for c, n := range base.Spec.Replicas {
		patch.Spec.Replicas[c] = n * multiplier
	}
	return yaml.Marshal(patch)
}

func dumpFiles(baseDir string, runtimeFilename string, m map[string][]byte) error {
	r := strings.NewReplacer("{runtime}", runtimeFilename)
	for p, content := range m {
		p = path.Join(baseDir, r.Replace(p))
		if err := ioutil.WriteFile(p, content, os.ModePerm); err != nil {
			return fmt.Errorf("cannot dump spec to file %q: %w", p, err)
		}
	}
	return nil
}

func (m *maker) makeExamplePatchFiles(
	basePath string,
	base *v1.Runtime,
	runtimeFilename string,
	env string,
	replicasMultiplier uint32,
) ([]byte, []byte, []byte, error) {
	devMode := "false"
	if env == "test" {
		devMode = "true"
	}
	iYAML, err := m.makePatchIYAML(basePath, runtimeFilename, env)
	if err != nil {
		return nil, nil, nil, fmt.Errorf("cannot generate i.yaml for env %q: %w", env, err)
	}
	patch, err := m.makeRuntimePatch(base, map[string]string{"DEBUG": devMode}, replicasMultiplier)
	if err != nil {
		return nil, nil, nil, fmt.Errorf("cannot generate runtime patch for env %q: %w", env, err)
	}
	config := []byte(fmt.Sprintf(exampleConfigTemplate, devMode))
	return iYAML, patch, config, nil
}

func makeDirs(dirs []string) error {
	for _, d := range dirs {
		if err := os.MkdirAll(d, os.ModePerm); err != nil {
			return fmt.Errorf("cannot create dir %q: %w", d, err)
		}
	}
	return nil
}

func (m *maker) dumpSpecs(arcRoot string, rt *v1.Runtime, substs *substitutio.Substitutions) (*specPaths, error) {
	// Let's put runtime to subdirectory with name equal to namespace name
	dir := m.nsName
	if _, err := os.Stat(dir); !errors.Is(err, os.ErrNotExist) {
		return nil, fmt.Errorf("directory %q already exists. Please remove it or enter another one", dir)
	}
	runtimeFilename := fmt.Sprintf("i.runtime.%v.yaml", rt.GetName())
	absDir, err := filepath.Abs(dir)
	if err != nil {
		return nil, fmt.Errorf("cannot find out abs path for given dir %q: %w", dir, err)
	}
	basePath, err := filepath.Rel(arcRoot, absDir)
	if err != nil {
		return nil, fmt.Errorf("cannot determine relative path from arc root to namespace dir %q: %w", absDir, err)
	}
	baseRuntime, err := yaml.Marshal(rt)
	if err != nil {
		return nil, fmt.Errorf("cannot marshal base template runtime to yaml: %w", err)

	}
	baseIYAML, err := m.makeBaseIYAML(runtimeFilename)
	if err != nil {
		return nil, fmt.Errorf("cannot marshal base i.yaml: %w", err)
	}
	testIYAML, testRuntime, testConfig, err := m.makeExamplePatchFiles(basePath, rt, runtimeFilename, "test", 2)
	if err != nil {
		return nil, fmt.Errorf("cannot generate specs for env \"test\": %w", err)
	}
	prodIYAML, prodRuntime, prodConfig, err := m.makeExamplePatchFiles(basePath, rt, runtimeFilename, "prod", 3)
	if err != nil {
		return nil, fmt.Errorf("cannot generate specs for env \"prod\": %w", err)
	}
	build, err := substs.DumpToString()
	if err != nil {
		return nil, fmt.Errorf("cannot dump default artifacts to .build.yaml file: %w", err)
	}
	buildDump := []byte(build)
	err = makeDirs([]string{dir, path.Join(dir, "test"), path.Join(dir, "prod")})
	if err != nil {
		return nil, err
	}
	err = dumpFiles(dir, runtimeFilename, map[string][]byte{
		"i.yaml":           baseIYAML,
		"{runtime}":        baseRuntime,
		"test/i.yaml":      testIYAML,
		"test/{runtime}":   testRuntime,
		"test/cfg.yml":     testConfig,
		"test/.build.yaml": buildDump,
		"prod/i.yaml":      prodIYAML,
		"prod/{runtime}":   prodRuntime,
		"prod/cfg.yml":     prodConfig,
		"prod/.build.yaml": buildDump,
	})
	if err != nil {
		return nil, err
	}
	return &specPaths{
		path.Join(absDir, "test"),
		path.Join(absDir, "prod"),
	}, nil
}
