package build

import (
	"bytes"
	"encoding/json"
	"fmt"
	"log"
	"os"
	"os/exec"
	"path"
	"strconv"

	"github.com/go-resty/resty/v2"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/infra/infractl/cli/commands/root"
	"a.yandex-team.ru/infra/infractl/cli/internal/arcutil"
	"a.yandex-team.ru/infra/infractl/cli/internal/serialization"
	substitutio "a.yandex-team.ru/infra/infractl/cli/internal/substitutions"
	dv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	rv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
)

func resolveSandboxResource(resourceID string) *substitutio.SandboxResource {
	client := resty.New()

	resourceInfo := SandboxResourceInfo{}
	taskInfo := SandboxTaskInfo{}

	_, err := client.R().
		SetHeader("Accept", "application/json; charset=utf-8").
		SetResult(&resourceInfo).
		Get(fmt.Sprintf("https://sandbox.yandex-team.ru/api/v1.0/resource/%v", resourceID))
	if err != nil {
		log.Fatalf("Failed to get sandbox resource %v info: %v", resourceID, err)
	}
	_, err = client.R().
		SetHeader("Accept", "application/json; charset=utf-8").
		SetResult(&taskInfo).
		Get(resourceInfo.Task.URL)
	if err != nil {
		log.Fatalf("Failed to get sandbox task %v info: %v", resourceInfo.Task.ID, err)
	}

	return &substitutio.SandboxResource{
		TaskType:     taskInfo.Type,
		TaskID:       strconv.FormatUint(taskInfo.ID, 10),
		ResourceType: resourceInfo.Type,
		ResourceID:   strconv.FormatUint(resourceInfo.ID, 10),
		Attributes:   resourceInfo.Attributes,
	}
}

func getPackages(filename string) (dockerPackages []string, sandboxPackages []string, err error) {
	fileSpecs, err := serialization.ParseYaml(filename)
	if err != nil {
		return
	}

	if len(fileSpecs) == 0 {
		log.Fatalf("File %q contains no known objects", filename)
	}

	for _, fileSpec := range fileSpecs {
		switch spec := fileSpec.(type) {
		case *rv1.Runtime:
			packagePath := substitutio.TryParseDockerImage(spec.Spec.GetImage())
			if packagePath != nil {
				dockerPackages = append(dockerPackages, *packagePath)
			}
			for _, layer := range spec.Spec.Layers {
				packagePath = substitutio.TryParseSandboxResource(layer.ResourceId)
				if packagePath != nil {
					sandboxPackages = append(sandboxPackages, *packagePath)
				}
			}

		case *dv1.DeployStage:
			buildManifestString := spec.GetAnnotations()[substitutio.BuildManifestAnnotation]
			if len(buildManifestString) == 0 {
				continue
			}
			manifest, err := substitutio.LoadBuildManifest(buildManifestString)
			if err != nil {
				log.Fatalf("Failed to load build manifest from file %q: %v", filename, err)
			}

			for _, dockerPackage := range manifest.DockerPackages {
				dockerPackages = append(dockerPackages, dockerPackage.Path)
			}
			for _, layer := range manifest.Layers {
				sandboxPackages = append(sandboxPackages, layer.Path)
			}
			for _, staticResource := range manifest.StaticResources {
				sandboxPackages = append(sandboxPackages, staticResource.Path)
			}
		}
	}
	return
}

func parsePackagesJSONImageSpec(path string) (image string, err error) {
	f, err := os.Open(path)
	if err != nil {
		return
	}

	var packagesJSON []struct {
		DockerImage string `json:"docker_image"`
		Digest      string `json:"digest"`
	}

	jsonDecoder := json.NewDecoder(f)
	err = jsonDecoder.Decode(&packagesJSON)
	if err != nil {
		return
	}
	if len(packagesJSON) != 1 {
		err = fmt.Errorf("currently 'ya package' must build exactly one package, %v found", len(packagesJSON))
		return
	}
	packageJSON := packagesJSON[0]
	if len(packageJSON.DockerImage) == 0 || len(packageJSON.Digest) == 0 {
		err = fmt.Errorf("cannot find either docker_image or digest in packages.json")
		return
	}
	image = packageJSON.DockerImage + "@" + packageJSON.Digest
	return
}

func parsePackagesJSONResourceSpec(path string) (resource *substitutio.SandboxResource, err error) {
	f, err := os.Open(path)
	if err != nil {
		return
	}

	var packagesJSON []struct {
		PackageResourceID uint64 `json:"package_resource_id"`
	}

	jsonDecoder := json.NewDecoder(f)
	err = jsonDecoder.Decode(&packagesJSON)
	if err != nil {
		return
	}
	if len(packagesJSON) != 1 {
		err = fmt.Errorf("currently 'ya package' must build exactly one package, %v found", len(packagesJSON))
		return
	}
	packageJSON := packagesJSON[0]
	if packageJSON.PackageResourceID == 0 {
		err = fmt.Errorf("cannot find resuorce id for package built in packages.json")
		return
	}
	resource = resolveSandboxResource(strconv.FormatUint(packageJSON.PackageResourceID, 10))
	return
}

func collectPackages(filename string, arcPath string, collectedImages map[string]string, collectedResources map[string]*substitutio.SandboxResource) error {
	dockerPackagesToBuild, sandboxPackagesToBuild, err := getPackages(filename)
	if err != nil {
		return err
	}
	var arcRoot, ya, builtImage string
	var builtResource *substitutio.SandboxResource
	for _, packageName := range dockerPackagesToBuild {
		if _, ok := collectedImages[packageName]; ok {
			continue
		}
		log.Printf("Will build package %q\n", packageName)
		arcRoot, ya, err = arcutil.ArcadiaRootAndYa(arcPath)
		if err != nil {
			return err
		}

		var stdout, stderr bytes.Buffer
		buildCmd := exec.Command(ya, "package", "--docker", "--docker-push", packageName)
		buildCmd.Dir = arcRoot
		buildCmd.Stdout = &stdout
		buildCmd.Stderr = &stderr

		err = buildCmd.Run()
		if err != nil {
			return err
		}
		builtImage, err = parsePackagesJSONImageSpec(path.Join(arcRoot, "packages.json"))
		if err != nil {
			return err
		}
		log.Printf("Collected docker image spec: %q", builtImage)
		collectedImages[packageName] = builtImage
	}
	for _, packageName := range sandboxPackagesToBuild {
		if _, ok := collectedResources[packageName]; ok {
			continue
		}
		log.Printf("Will build package %q\n", packageName)

		arcRoot, ya, err = arcutil.ArcadiaRootAndYa(arcPath)
		if err != nil {
			return err
		}

		var stdout, stderr bytes.Buffer
		// TODO support resource type in manifest?
		buildCmd := exec.Command(
			ya, "package", "--tar", "--upload", "--sandbox",
			"--upload-resource-attr", "package_path="+packageName,
			"--target-platform", "DEFAULT-LINUX-X86_64",
			packageName,
		)
		buildCmd.Dir = arcRoot
		buildCmd.Stdout = &stdout
		buildCmd.Stderr = &stderr

		err = buildCmd.Run()
		if err != nil {
			return err
		}

		builtResource, err = parsePackagesJSONResourceSpec(path.Join(arcRoot, "packages.json"))
		if err != nil {
			return err
		}
		collectedResources[packageName] = builtResource
	}
	return nil
}

func Build() *cobra.Command {
	substs := substitutio.NewSubstitutions()
	var outputFile string
	var arcPath string
	var incremental bool
	var commandLineDockerPackages map[string]string
	var commandLineSandboxResources map[string]string
	cmd := &cobra.Command{
		Use:   "build",
		Short: "Build all required artifacts and prepare substitutions file",
		Args:  root.ValidFile,
		Run: func(cmd *cobra.Command, args []string) {
			if incremental {
				substs = substitutio.Load(outputFile)
			}
			for k, v := range commandLineDockerPackages {
				substs.DockerPackages[k] = v
			}
			for k, v := range commandLineSandboxResources {
				substs.SandboxResources[k] = resolveSandboxResource(v)
			}
			for _, filename := range args {
				err := collectPackages(filename, arcPath, substs.DockerPackages, substs.SandboxResources)
				if err != nil {
					log.Fatalf("Failed to process file %q: %v", filename, err)
				}
			}
			err := substs.DumpToFile(outputFile)
			if err != nil {
				stringVal, err2 := substs.DumpToString()
				if err2 == nil {
					log.Fatalf("Failed to write substitutions to %q: %v\n\nCollected data:\n\n%v", outputFile, err, stringVal)
				} else {
					log.Fatalf("Failed to write substitutions to %q: %v\n\nCollected data couldn't be dumped: %v", outputFile, err, err2)
				}
			}
		},
	}
	cmd.Flags().StringVarP(&arcPath, "arc-path", "a", ".", "arcadia root path to find artifacts (package.json, etc) files")
	cmd.Flags().StringToStringVarP(&commandLineDockerPackages, "docker", "d", map[string]string{}, "add prebuilt path/to/package.json=docker/image:tag@sha")
	cmd.Flags().StringToStringVarP(&commandLineSandboxResources, "sandbox", "s", map[string]string{}, "add prebuilt sandbox resource path/to/package.json=1234567890")
	cmd.Flags().StringVarP(&outputFile, "output", "o", ".build.yaml", "file path to write substitutions into")
	cmd.Flags().BoolVarP(&incremental, "incremental", "i", false, "update file contents with new values rather than rewrite")
	return cmd
}
