package create

import (
	"fmt"
	"log"

	"github.com/spf13/cobra"
	corev1 "k8s.io/api/core/v1"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/cli/commands/root"
	"a.yandex-team.ru/infra/infractl/cli/internal/serialization"
	"a.yandex-team.ru/infra/infractl/internal/labels"
	"a.yandex-team.ru/infra/infractl/util/kubeutil"
)

const (
	nameLabel = "kubernetes.io/metadata.name"
)

func CreateNamespace() *cobra.Command {
	var abcName, outputFile string
	var push bool

	cmd := &cobra.Command{
		Use:     "namespace NAME",
		Short:   "Create namespace spec",
		Args:    cobra.ExactArgs(1),
		Aliases: []string{"ns"},
		Run: func(cmd *cobra.Command, args []string) {
			nsName := args[0]

			if len(abcName) == 0 {
				log.Fatal("`--abc SLUG` is required. Each namespace must be bound to existing ABC service")
			}

			ns := &corev1.Namespace{}

			// FIXME where can we properly take GVK?
			ns.APIVersion = corev1.SchemeGroupVersion.String()
			ns.Kind = "Namespace"

			ns.Name = nsName
			ns.Labels = map[string]string{
				labels.ABC: abcName,
				nameLabel:  nsName,
			}

			var obj client.Object = ns

			serialization.MustDumpObject(obj, outputFile)

			c := kubeutil.MakeClient()
			objectKey := client.ObjectKeyFromObject(obj)
			if push {
				if _, err := c.PutObject(root.Context, objectKey, obj); err != nil {
					log.Fatalf("Failed to push object to k8s: %v", err)
				}
				fmt.Printf("Namespace %q has been pushed successfully\n", ns.Name)
			}
		},
	}
	cmd.Flags().StringVarP(&abcName, "abc", "a", "", "ABC service namespace would be bound to")
	cmd.Flags().StringVarP(&outputFile, "output", "o", "-", "File to dump generated spec into")
	cmd.Flags().BoolVar(&push, "push", false, "If set, spec will be pushed to k8s after creation")

	return cmd
}
