package cmdimport

import (
	"fmt"
	"log"

	"github.com/spf13/cobra"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"

	"a.yandex-team.ru/infra/awacs/clients/go/awacs"
	"a.yandex-team.ru/infra/infractl/cli/commands/root"
	"a.yandex-team.ru/infra/infractl/cli/internal/serialization"
	"a.yandex-team.ru/infra/infractl/controllers/awacs/api/backend/proto_v1"
	v1 "a.yandex-team.ru/infra/infractl/controllers/awacs/api/backend/v1"
	"a.yandex-team.ru/infra/infractl/util/awacsutil"
	"a.yandex-team.ru/infra/infractl/util/oauthutil"
)

func ImportAwacsBackend() *cobra.Command {
	var outputFile, namespace, awacsToken string
	cmd := &cobra.Command{
		Use:   "awacsbackend AWACS_NAMESPACE BACKEND_NAME",
		Short: "Import existing awacs backend",
		Args:  cobra.ExactArgs(2),
		Run: func(cmd *cobra.Command, args []string) {
			if len(awacsToken) == 0 {
				var err error
				awacsToken, err = oauthutil.GetToken(root.Context, "AWACS_TOKEN")
				if err != nil {
					log.Fatalf("Awacs access token is not given and cannot be retrieved by ssh\n" +
						"Please get you AWACS access token from: " +
						"https://oauth.yandex-team.ru/authorize?response_type=code&client_id=447a6c29a65345f0a4d31784bb850bbd " +
						"and put it into AWACS_TOKEN env variable")
				}
			}

			awacsClient := awacs.NewClient(awacs.WithToken(awacsToken))
			backend, err := awacsutil.GetBackend(root.Context, awacsClient, args[0], args[1])
			if err != nil {
				log.Fatalf("Failed to get backend from awacs: %v", err)
			}
			if backend == nil {
				log.Fatal("Backend not found in awacs")
			}

			kBackend := &v1.AwacsBackend{
				ObjectMeta: metav1.ObjectMeta{
					Name:        fmt.Sprintf("%v--%v", args[0], args[1]),
					Namespace:   namespace,
					Labels:      map[string]string{},
					Annotations: map[string]string{},
				},
				TypeMeta: v1.AwacsBackendTypeMeta,
				Spec: &proto_v1.Spec{
					BackendSpec: backend.Spec,
				},
			}
			serialization.MustDumpObject(kBackend, outputFile)
		},
		SilenceUsage:  true,
		SilenceErrors: true,
	}

	cmd.Flags().StringVarP(&namespace, "namespace", "n", "", "K8s namespace to assign imported object into")
	cmd.Flags().StringVarP(&outputFile, "output", "o", "-", "File to dump spec into")
	cmd.Flags().StringVar(&awacsToken, "awacs-token", "", "awacs token value. Environment variable AWACS_TOKEN is recommended instead")
	return cmd
}
