package cmdimport

import (
	"log"

	"github.com/spf13/cobra"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"

	"a.yandex-team.ru/infra/infractl/cli/commands/root"
	yputil "a.yandex-team.ru/infra/infractl/cli/commands/util/yp"
	"a.yandex-team.ru/infra/infractl/cli/internal/serialization"
	pdv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/project/proto_v1"
	dv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/project/v1"
	"a.yandex-team.ru/infra/infractl/util/oauthutil"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yp/go/yp"
)

func ImportProject() *cobra.Command {
	var outputFile, ypToken string
	var ypCluster = yputil.XDC

	cmd := &cobra.Command{
		Use:   "project PROJECT_NAME",
		Short: "Import existing Yandex.Deploy project",
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			if len(ypToken) == 0 {
				var err error
				ypToken, err = oauthutil.GetToken(root.Context, "YP_TOKEN")
				if err != nil {
					log.Fatalf("YP access token is not given\n" +
						"Please get your YP access token from: " +
						"https://oauth.yandex-team.ru/authorize?response_type=token&client_id=3e5a6e634dd243088d230775c401147d " +
						"and put it into YP_TOKEN env variable")
				}
			}

			ypClient, err := yp.NewClient(ypCluster.String(), yp.WithAuthToken(ypToken))
			if err != nil {
				log.Fatalf("Failed to create YP connection: %v", err)
			}

			kProject := dv1.DeployProject{
				TypeMeta: dv1.DeployProjectTypeMeta,
				ObjectMeta: metav1.ObjectMeta{
					Name:        "",
					Labels:      map[string]string{},
					Annotations: map[string]string{},
				},
				Spec: &pdv1.Spec{
					ProjectSpec: &ypapi.TProjectSpec{},
				},
			}
			response, err := ypClient.GetObject(root.Context, yp.GetObjectRequest{
				ObjectType: ypapi.EObjectType_OT_PROJECT,
				ObjectID:   args[0],
				Selectors:  []string{"/meta", "/spec"},
				Format:     ypapi.EPayloadFormat_PF_PROTOBUF,
			})
			if err != nil {
				log.Fatalf("Failed to get project from YP: %v", err)
			}

			projectMeta := &ypapi.TProjectMeta{}
			err = response.Fill(projectMeta, kProject.Spec.ProjectSpec)
			if err != nil {
				log.Fatalf("Failed to parse project from YP: %v", err)
			}
			kProject.Name = projectMeta.Id
			kProject.Namespace = projectMeta.Id
			kProject.Annotations[dv1.FqidAnnotation] = projectMeta.Fqid
			// drop not meaningful fields to reduce the listing size

			kProject.Status = nil
			kProject.Spec.ProjectSpec.AccountId = ""
			serialization.MustDumpObject(&kProject, outputFile)
		},
		SilenceUsage:  true,
		SilenceErrors: true,
	}
	cmd.Flags().StringVarP(&outputFile, "output", "o", "-", "File to dump spec into")
	cmd.Flags().StringVar(&ypToken, "yp-token", "", "YP token value. Environment variable YP_TOKEN is recommended instead")
	cmd.Flags().Var(&ypCluster, "cluster", "YP cluster to import from.")

	return cmd
}
