package cmdimport

import (
	"log"

	"github.com/spf13/cobra"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"

	"a.yandex-team.ru/infra/infractl/cli/commands/root"
	yputil "a.yandex-team.ru/infra/infractl/cli/commands/util/yp"
	"a.yandex-team.ru/infra/infractl/cli/internal/serialization"
	pdv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/proto_v1"
	dv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yp/go/yp"
)

func ImportStage() *cobra.Command {
	var outputFile, ypToken string
	var ypCluster = yputil.XDC

	cmd := &cobra.Command{
		Use:   "stage STAGE_NAME",
		Short: "Import existing Yandex.Deploy stage",
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			ypClient := yputil.GetYPClientOrFail(ypCluster.String(), ypToken, "YP_TOKEN")

			kStage := dv1.DeployStage{
				TypeMeta: dv1.DeployStageTypeMeta,
				ObjectMeta: metav1.ObjectMeta{
					Name:        "",
					Labels:      map[string]string{},
					Annotations: map[string]string{},
				},
				Spec: &pdv1.Spec{
					StageSpec: &ypapi.TStageSpec{},
				},
			}
			response, err := ypClient.GetObject(root.Context, yp.GetObjectRequest{
				ObjectType: ypapi.EObjectType_OT_STAGE,
				ObjectID:   args[0],
				Selectors:  []string{"/meta", "/spec"},
				Format:     ypapi.EPayloadFormat_PF_PROTOBUF,
			})
			if err != nil {
				log.Fatalf("Failed to get stage from YP: %v", err)
			}

			stageMeta := &ypapi.TStageMeta{}
			err = response.Fill(stageMeta, kStage.Spec.StageSpec)
			if err != nil {
				log.Fatalf("Failed to parse stage from YP: %v", err)
			}
			kStage.Name = stageMeta.Id
			kStage.Namespace = stageMeta.ProjectId
			kStage.Annotations[dv1.FqidAnnotation] = stageMeta.Fqid
			// drop not meaningful fields to reduce the listing size
			kStage.Status = nil
			kStage.Spec.StageSpec.RevisionInfo = nil
			kStage.Spec.StageSpec.Revision = 0
			kStage.Spec.StageSpec.AccountId = ""
			for _, du := range kStage.Spec.StageSpec.DeployUnits {
				du.PatchersRevision = 0
				du.Revision = 0
				du.PodAgentSandboxInfo = nil
				du.PodAgentLayerSandboxInfo = nil
				du.DynamicResourceUpdaterSandboxInfo = nil
				du.LogbrokerToolsSandboxInfo = nil
				du.TvmSandboxInfo = nil
			}

			serialization.MustDumpObject(&kStage, outputFile)
		},
		SilenceUsage:  true,
		SilenceErrors: true,
	}
	cmd.Flags().StringVarP(&outputFile, "output", "o", "-", "File to dump spec into")
	cmd.Flags().StringVar(&ypToken, "yp-token", "", "YP token value. Environment variable YP_TOKEN is recommended instead")
	cmd.Flags().Var(&ypCluster, "cluster", "YP cluster to import from.")

	return cmd
}
