package kubeconfig

import (
	"fmt"
	"io/ioutil"
	"log"
	"os"

	"github.com/spf13/cobra"
	"k8s.io/client-go/tools/clientcmd"

	"a.yandex-team.ru/infra/infractl/cli/commands/root"
	"a.yandex-team.ru/infra/infractl/internal/environs"
	"a.yandex-team.ru/infra/infractl/internal/kubeconfig"
	"a.yandex-team.ru/infra/infractl/util/kubeutil"
)

func SetupKubeConfig() *cobra.Command {
	var stdout bool
	var extraContext string

	cmd := &cobra.Command{
		Use:   "setup",
		Short: "Merge default kubeconfig (if exists) with InfraCtl kubeconfig",
		Run: func(cmd *cobra.Command, args []string) {
			kubeContexts := make([]string, 0, 2)
			kubeContexts = append(kubeContexts, environs.ProdEnviron.ConfigOpts.Context)
			aliasesMap := getContextAliasesMap()
			extraContext = aliasesMap[extraContext]
			if extraContext != "" && extraContext != environs.ProdEnviron.ConfigOpts.Context {
				kubeContexts = append(kubeContexts, extraContext)
			}
			infraCtlKubeConfig := kubeutil.MakeInfraCtlKubeConfig(root.Context, kubeContexts)
			configFile, _ := ioutil.TempFile("", "")
			defer os.Remove(configFile.Name())

			if err := clientcmd.WriteToFile(*infraCtlKubeConfig, configFile.Name()); err != nil {
				log.Fatalf("Failed to save InfraCtl kubeconfig to temporary file %s: %s", configFile.Name(), err)
			}
			rules := clientcmd.ClientConfigLoadingRules{
				Precedence: []string{kubeconfig.Path, configFile.Name()},
			}
			mergedConf, err := rules.Load()
			if err != nil {
				log.Fatalf("Failed to generate kubeconfig: %s", err)
			}
			if stdout {
				d, err := clientcmd.Write(*mergedConf)
				if err != nil {
					log.Fatalf("Failed to encode kubeconfig: %s", err)
				}
				fmt.Println(string(d))
			} else {
				if err = clientcmd.WriteToFile(*mergedConf, kubeconfig.Path); err != nil {
					log.Fatalf("Failed to save kubeconfig to file %s: %s", kubeconfig.Path, err)
				}
				fmt.Printf("Kubeconfig suceessfully saved to %s file\n", kubeconfig.Path)
			}
		},
	}

	cmd.Flags().BoolVar(&stdout, "stdout", false, "If set, kubeconfig will be printed to stdout, otherwise it will be saved by --kubeconfig path (default is $HOME/.kube/config)")
	cmd.Flags().StringVar(&extraContext, "context", "", "Add extra context to kubeconfig. For now only dev-k supported")
	return cmd
}
