package kubeconfig

import (
	"log"

	"github.com/spf13/cobra"
	"k8s.io/client-go/tools/clientcmd"
	clientcmdapi "k8s.io/client-go/tools/clientcmd/api"

	"a.yandex-team.ru/infra/infractl/internal/kubeconfig"
)

func validate(config *clientcmdapi.Config, contextName string) {
	if len(contextName) == 0 {
		log.Fatal("Empty context names are not allowed")
	}

	for name := range config.Contexts {
		if name == contextName {
			return
		}
	}

	log.Fatalf("No context exists with the name: %q", contextName)
}

func UseContext() *cobra.Command {
	cmd := &cobra.Command{
		Use:     "use-context CONTEXT_NAME",
		Args:    cobra.ExactArgs(1),
		Aliases: []string{"use"},
		Short:   "Set the current-context in a kubeconfig file",
		Run: func(cmd *cobra.Command, args []string) {
			aliasesMap := getContextAliasesMap()
			name := args[0]
			contextName, ok := aliasesMap[name]
			if !ok {
				contextName = name
			}
			config, err := clientcmd.NewNonInteractiveDeferredLoadingClientConfig(
				&clientcmd.ClientConfigLoadingRules{ExplicitPath: kubeconfig.Path},
				&clientcmd.ConfigOverrides{}).RawConfig()
			validate(&config, contextName)
			if err != nil {
				log.Fatalf("Failed to read kubeconfig: %v", err)
			}
			config.CurrentContext = contextName
			if err = clientcmd.WriteToFile(config, kubeconfig.Path); err != nil {
				log.Fatalf("Failed to save kubeconfig to file %s: %s", kubeconfig.Path, err)
			}
		},
	}
	return cmd
}
