package modify

import (
	"fmt"
	"log"
	"os"

	"github.com/spf13/cobra"
	corev1 "k8s.io/api/core/v1"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/cli/commands/root"
	"a.yandex-team.ru/infra/infractl/cli/internal/serialization"
	"a.yandex-team.ru/infra/infractl/internal/secrets"
	"a.yandex-team.ru/infra/infractl/util/kubeutil"
	"a.yandex-team.ru/infra/infractl/util/oauthutil"
)

func Delegate(filename *string) *cobra.Command {
	cmd := &cobra.Command{
		Use:              "delegate",
		Short:            "Delegate token and add it to the namespace",
		TraverseChildren: true,
	}

	cmd.AddCommand(DelegateAwacsToken(filename))
	cmd.AddCommand(DelegateYPToken(filename))

	return cmd
}

type DelegateToken struct {
	token            string
	filename         *string
	provider         string
	envName          string
	argName          string
	url              string
	tvmAliasesAndIDs []string
}

func (d *DelegateToken) Run(_ *cobra.Command, _ []string) {
	if len(d.token) == 0 {
		d.token = os.Getenv(d.envName)
	}
	if len(d.token) == 0 {
		log.Fatalf("--%v parameter or %v env var must be set to delegate secret.\n"+
			"You can get (please, use robot account) access token from: %v ", d.argName, d.envName, d.url)
	}
	if len(*d.filename) == 0 {
		log.Fatal("Namespace --filename must be given")
	}

	yavToken := oauthutil.MustGetToken(root.Context, "YAV_TOKEN")
	yavClient := secrets.MustCreateClient(yavToken)
	kubeClient := kubeutil.MakeClient()

	secretUUID, err := ensureYavSecret(&kubeClient, yavClient, d.provider, *d.filename, d.tvmAliasesAndIDs)
	if err != nil {
		log.Fatal(err)
	}
	if err := yavClient.SetSecretField(root.Context, secretUUID, secrets.CommonTokenYavKey, d.token); err != nil {
		log.Fatalf("Writing %v secret failed: %v", d.provider, err)
	}
}

func (d *DelegateToken) ArgDescription() string {
	return fmt.Sprintf("%v token, alternatively can be set through %v env variable", d.provider, d.envName)
}

func DelegateYPToken(filename *string) *cobra.Command {
	runner := DelegateToken{
		filename:         filename,
		provider:         "yp",
		envName:          "YP_TOKEN",
		argName:          "yp-token",
		url:              "https://oauth.yandex-team.ru/authorize?response_type=token&client_id=3e5a6e634dd243088d230775c401147d",
		tvmAliasesAndIDs: []string{"deployctl"},
	}
	cmd := &cobra.Command{
		Use:   "yp",
		Short: "Delegate YP token and add it to the namespace",
		Run:   runner.Run,
	}
	cmd.Flags().StringVar(&runner.token, runner.argName, "", runner.ArgDescription())
	cmd.Flags().StringSliceVar(&runner.tvmAliasesAndIDs, "to", runner.tvmAliasesAndIDs, "TVM application IDs allowed to read secret")
	return cmd
}

func DelegateAwacsToken(filename *string) *cobra.Command {
	runner := DelegateToken{
		filename:         filename,
		provider:         "awacs",
		envName:          "AWACS_TOKEN",
		argName:          "awacs-token",
		url:              "https://oauth.yandex-team.ru/authorize?response_type=code&client_id=447a6c29a65345f0a4d31784bb850bbd",
		tvmAliasesAndIDs: []string{"awacsctl"},
	}
	cmd := &cobra.Command{
		Use:   "awacs",
		Short: "Delegate awacs token and add it to the namespace",
		Run:   runner.Run,
	}
	cmd.Flags().StringVar(&runner.token, runner.argName, "", runner.ArgDescription())
	cmd.Flags().StringSliceVar(&runner.tvmAliasesAndIDs, "to", runner.tvmAliasesAndIDs, "TVM application IDs allowed to read secret")
	return cmd
}

func ensureYavSecret(
	kubeClient *kubeutil.Client,
	yavClient secrets.YavClient,
	provider, filename string,
	tvmAliasesAndIDs []string,
) (string, error) {
	obj := serialization.MustReadOneObject(filename)
	if kind := obj.GetObjectKind().GroupVersionKind().Kind; kind != "Namespace" {
		return "", fmt.Errorf("invalid object kind in file %q, Namespace expected, found: %q", filename, kind)
	}

	ns := obj.(*corev1.Namespace)
	if err := kubeClient.Get(root.Context, client.ObjectKey{Name: ns.Name}, ns); err != nil {
		return "", err
	}
	// why do we need to do it at all?
	ns.APIVersion = corev1.SchemeGroupVersion.String()
	ns.Kind = "Namespace"

	return secrets.EnsureSecretForNamespace(root.Context, secrets.EnsureSecretOptions{
		KubeClient:       kubeClient,
		YAVClient:        &yavClient,
		Provider:         provider,
		Namespace:        ns,
		TVMAliasesAndIDs: tvmAliasesAndIDs,
	})
}
