package modify

import (
	"fmt"
	"log"

	"github.com/ghodss/yaml"
	"github.com/spf13/cobra"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/cli/internal/deploystage"
	"a.yandex-team.ru/infra/infractl/cli/internal/serialization"
	substitutio "a.yandex-team.ru/infra/infractl/cli/internal/substitutions"
	dv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
)

func SetTemplate(filename *string) *cobra.Command {
	cmd := &cobra.Command{
		Use:              "set-template",
		Aliases:          []string{"set", "add-template"},
		Short:            "Replace fixed artifact with build manifest template",
		TraverseChildren: true,
	}

	cmd.AddCommand(SetDockerPackage(filename))
	cmd.AddCommand(SetSandboxResource(filename, &deploystage.Layer{}, &substitutio.LayerFiller{}, "layer"))
	cmd.AddCommand(SetSandboxResource(filename, &deploystage.StaticResource{}, &substitutio.StaticResourceFiller{}, "static-resource"))

	return cmd
}

func SetSandboxResource(filename *string, sandboxRsc deploystage.Resource, resourceWrapper substitutio.ManifestFiller, cmdFlag string) *cobra.Command {
	var duID, resourceID, outputFile string
	cmd := &cobra.Command{
		Use:   cmdFlag,
		Short: fmt.Sprintf("Replace %s with reference to `ya package` build target", sandboxRsc.GetResourceName()),
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			obj := loadDeployStage(*filename)
			resolvedDuID, du, err := deploystage.DeduceDeployUnit(obj.Spec.StageSpec, duID)
			if err != nil {
				log.Fatalf("Failed to select deploy unit: %v", err)
			}
			resolvedSandboxResourceID, sandboxResource, err := deploystage.DeduceResource(du, resourceID, sandboxRsc)
			if err != nil {
				log.Fatalf("Failed to select %s: %v", sandboxRsc.GetResourceName(), err)
			}

			substitutions := substitutio.Load(outputFile)

			resource := sandboxResource.GetSandboxResource()
			attrs := make(map[string]interface{}, len(resource.GetAttributes()))
			for k, v := range resource.GetAttributes() {
				attrs[k] = v
			}
			substitutions.SandboxResources[args[0]] = &substitutio.SandboxResource{
				TaskType:     resource.GetTaskType(),
				TaskID:       resource.GetTaskId(),
				ResourceType: resource.GetResourceType(),
				ResourceID:   resource.GetResourceId(),
				Attributes:   attrs,
			}

			if err := substitutions.DumpToFile(outputFile); err != nil {
				log.Fatalf("Falied to save current build: %v", err)
			}

			sandboxResource.SetDefaultFieldsNull()
			manifest := loadBuildManifest(obj)

			resourceWrapper.FillManifest(manifest, duID, resolvedDuID, resolvedSandboxResourceID, resourceID, args[0])

			dump, err := yaml.Marshal(manifest)
			if err != nil {
				log.Fatalf("Failed to marshal updated build manifest: %v", err)
			}

			obj.Annotations[substitutio.BuildManifestAnnotation] = string(dump)

			err = serialization.DumpToFile(obj, *filename)
			if err != nil {
				log.Fatalf("Failed to marshal updated object: %v\nWill dump to stderr:\n\n%v\n\n", err, string(dump))
			}
		},
	}
	cmd.Flags().StringVar(&duID, "du", "", "deploy unit to modify")
	cmd.Flags().StringVar(&resourceID, cmdFlag, "", sandboxRsc.GetResourceName()+" to replace")
	cmd.Flags().StringVarP(&outputFile, "output", "o", ".build.yaml", "file path to write substitutions into")
	return cmd
}

func SetDockerPackage(filename *string) *cobra.Command {
	var duID, boxID, outputFile string
	cmd := &cobra.Command{
		Use:   "docker",
		Short: "Replace docker image with reference to `ya package` build target",
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			obj := loadDeployStage(*filename)
			resolvedDuID, du, err := deploystage.DeduceDeployUnit(obj.Spec.StageSpec, duID)
			if err != nil {
				log.Fatalf("Failed to select deploy unit: %v", err)
			}

			resolvedBoxID, _, err := deploystage.DeduceBox(du, boxID)
			if err != nil {
				log.Fatalf("Failed to select box: %v", err)
			}

			substitutions := substitutio.Load(outputFile)

			image := du.ImagesForBoxes[resolvedBoxID]
			imageString := fmt.Sprintf("%s/%s:%s@%s", image.RegistryHost, image.Name, image.Tag, image.Digest)
			substitutions.DockerPackages[args[0]] = imageString

			if err := substitutions.DumpToFile(outputFile); err != nil {
				log.Fatalf("Falied to save current build: %v", err)
			}

			delete(du.ImagesForBoxes, resolvedBoxID)
			if obj.Annotations == nil {
				obj.Annotations = map[string]string{}
			}

			manifest := loadBuildManifest(obj)

			for i, dockerPackage := range manifest.DockerPackages {
				if (dockerPackage.DeployUnitID == duID || dockerPackage.DeployUnitID == resolvedDuID) &&
					(dockerPackage.BoxID == boxID || dockerPackage.BoxID == resolvedBoxID) {
					manifest.DockerPackages = append(manifest.DockerPackages[:i], manifest.DockerPackages[i+1:]...)
					break
				}
			}

			manifest.DockerPackages = append(manifest.DockerPackages, substitutio.DockerPackageRef{
				DeployUnitID: duID,
				BoxID:        boxID,
				Path:         args[0],
			})

			dump, err := yaml.Marshal(manifest)
			if err != nil {
				log.Fatalf("Failed to marshal updated build manifest: %v", err)
			}

			obj.Annotations[substitutio.BuildManifestAnnotation] = string(dump)

			err = serialization.DumpToFile(obj, *filename)
			if err != nil {
				log.Fatalf("Failed to marshal updated object: %v\nWill dump to stderr:\n\n%v\n\n", err, string(dump))
			}
		},
	}
	cmd.Flags().StringVar(&duID, "du", "", "deploy unit to modify")
	cmd.Flags().StringVar(&boxID, "box", "", "box to modify")
	cmd.Flags().StringVarP(&outputFile, "output", "o", ".build.yaml", "file path to write substitutions into")

	return cmd
}

func loadDeployStage(filename string) *dv1.DeployStage {
	fileSpecs, err := serialization.ParseYaml(filename)
	if err != nil {
		log.Fatalf("Failed to read spec from %q: %v", filename, err)
	}
	if len(fileSpecs) != 1 {
		log.Fatalf("Currently only files with exactly one object supported, %v found", len(fileSpecs))
	}

	spec := fileSpecs[0]
	kind := spec.GetObjectKind().GroupVersionKind()
	if kind != dv1.GroupVersion.WithKind("DeployStage") {
		log.Fatalf("This command operates only with DeployStage objects, %q found", kind.Kind)
	}

	return spec.(*dv1.DeployStage)
}

func loadBuildManifest(obj client.Object) *substitutio.StageBuildManifest {
	if obj.GetAnnotations() == nil {
		obj.SetAnnotations(map[string]string{})
	}

	buildManifestString := obj.GetAnnotations()[substitutio.BuildManifestAnnotation]
	if len(buildManifestString) == 0 {
		return &substitutio.StageBuildManifest{}
	} else {
		manifest, err := substitutio.LoadBuildManifest(buildManifestString)
		if err != nil {
			log.Fatalf("Build manifest annotation is corrupted, manual editing is needed: %v", err)
		}
		return manifest
	}
}
