package pull

import (
	"fmt"
	"io/ioutil"
	"log"
	"strings"

	"github.com/spf13/cobra"
	"k8s.io/apimachinery/pkg/runtime/schema"
	utilruntime "k8s.io/apimachinery/pkg/util/runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/cli/commands/root"
	"a.yandex-team.ru/infra/infractl/cli/internal/serialization"
	substitutio "a.yandex-team.ru/infra/infractl/cli/internal/substitutions"
	"a.yandex-team.ru/infra/infractl/util/kubeutil"
)

type resource struct {
	schema.GroupVersionKind
	string
}

func parseObjects(client kubeutil.Client, objects []string) map[resource]bool {
	if len(objects) == 0 {
		log.Fatalf("Empty objects list given")
	}

	output := map[resource]bool{}

	typeCache := map[string]schema.GroupVersionKind{}

	for _, object := range objects {
		nameTuple := strings.Split(object, "/")
		if len(nameTuple) != 2 {
			log.Fatalf("Invalid object name %q: must be in form \"kind/name\"", object)
		}
		kind := nameTuple[0]
		name := nameTuple[1]
		if gvk, ok := typeCache[kind]; ok {
			output[resource{gvk, name}] = true
			continue
		}
		// TODO (torkve) to also resolve shortnames we have to find how to access discovery client
		gvk, err := client.ResolveKind(kind)
		if err != nil {
			log.Fatalf("Can't parse kind %q: %v", kind, err)
		}
		typeCache[kind] = gvk
		output[resource{gvk, name}] = true
	}
	return output
}

func Pull() *cobra.Command {
	strict := false
	globalFilename := ""

	cmd := &cobra.Command{
		Use:   "pull {namespace} {kind/name}...",
		Short: "Pull specs from k8s",
		Args:  cobra.MinimumNArgs(2),
		Run: func(cmd *cobra.Command, args []string) {
			namespace := args[0]

			substs := substitutio.NewSubstitutions()
			globalData := make([]byte, 0)
			objectsLoaded := 0

			c := kubeutil.MakeClient()

			for res := range parseObjects(c, args[1:]) {
				gvk := res.GroupVersionKind
				name := res.string

				objectKey := client.ObjectKey{
					Namespace: namespace,
					Name:      name,
				}
				runtimeObj, err := c.Scheme().New(gvk)
				if err != nil {
					log.Fatalf("Cannot create object of type %v: %v", gvk, err)
				}
				clientObj, ok := runtimeObj.(client.Object)
				if !ok {
					log.Fatalf("Cannot create object of type %v", gvk)
				}
				if err = c.Get(root.Context, objectKey, clientObj); err != nil {
					log.Fatalf("Object %v/%v not found: %v", gvk.Kind, name, err)
				}
				clientObj.SetManagedFields(nil)
				clientObj.GetObjectKind().SetGroupVersionKind(gvk)

				annotations := clientObj.GetAnnotations()
				var template []byte
				if annotations == nil || len(annotations[substitutio.TemplateAnnotation]) == 0 {
					log.Printf(
						"WARN: object %v/%v has no stored template, will use the object itself\n",
						gvk.Kind,
						name,
					)
					template, err = substitutio.DumpTemplate(clientObj)
					if err != nil {
						log.Fatalf("Failed to create template for %v/%v: %v", gvk.Kind, name, err)
					}
				} else {
					template = []byte(annotations[substitutio.TemplateAnnotation])
				}
				filename := fmt.Sprintf("infra.%v.%v.%v.yaml", namespace, gvk.Kind, name)
				objectsLoaded += 1

				if substsString := annotations[substitutio.SubstitutionsAnnotation]; len(substsString) != 0 {
					objSubsts, err := substitutio.LoadSubstitutions(substsString)
					if err != nil {
						log.Printf("%v %v substitutions are corrupted, ignoring: %v\n", gvk.Kind, name, err)
					} else {
						// TODO may be we should request some choice from user in non-strict mode?
						if err = substs.Merge(objSubsts, strict); err != nil {
							log.Fatalf("Failed to load substitutions from %v %v: %v", gvk.Kind, name, err)
						}
					}
				}
				if len(globalFilename) == 0 {
					fmt.Printf("Dumping %v\n", filename)
					utilruntime.Must(ioutil.WriteFile(filename, template, 0o600))
				} else {
					if objectsLoaded > 1 {
						globalData = append(globalData, []byte(serialization.YamlDocumentSeparator)...)
					}
					globalData = append(globalData, template...)
				}
			}
			if len(globalFilename) > 0 {
				fmt.Printf("Dumping %v\n", globalFilename)
				utilruntime.Must(ioutil.WriteFile(globalFilename, globalData, 0o600))
			}

			fmt.Printf("Dumping .build.yaml\n")
			utilruntime.Must(substs.DumpToFile(".build.yaml"))
		},
	}
	cmd.Flags().BoolVarP(&strict, "strict", "s", false,
		"Fail when conflicting substitutions found. Will use last encountered otherwise.")
	cmd.Flags().StringVarP(&globalFilename, "output", "o", "",
		"Filename to put all loaded specs into. If not specified, separate file will be used for each object")
	return cmd
}
