package put

import (
	"fmt"
	"log"
	"os"
	"sort"
	"time"

	"github.com/spf13/cobra"
	v1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/types"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/cli/commands/root"
	"a.yandex-team.ru/infra/infractl/cli/internal/arcutil"
	"a.yandex-team.ru/infra/infractl/cli/internal/deployedcheckers"
	"a.yandex-team.ru/infra/infractl/cli/internal/serialization"
	substitutio "a.yandex-team.ru/infra/infractl/cli/internal/substitutions"
	abv1 "a.yandex-team.ru/infra/infractl/controllers/awacs/api/backend/v1"
	auv1 "a.yandex-team.ru/infra/infractl/controllers/awacs/api/upstream/v1"
	dv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	rv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	"a.yandex-team.ru/infra/infractl/util/conditionchecker"
	"a.yandex-team.ru/infra/infractl/util/kubeutil"
	math2 "a.yandex-team.ru/library/go/x/math"
)

const defaultWaitTimeout = 0 // by default block infinitely waiting for put

var kindsOrder = map[string]int{
	"DeployStage": 1,
	"Runtime":     2,
	"Namespace":   3,
}

var substitutionsPath string
var revisionComment string
var shareSecrets bool
var waitDeployed bool
var waitDeployedTimeout int

func makeArcRevisionMsg() string {
	var revMsg string
	arcInfo, err := arcutil.ArcInfo()
	if err != nil {
		return ""
	}
	if arcInfo.Revision != 0 {
		revMsg = fmt.Sprintf("arc revision %d", arcInfo.Revision)
	} else if arcInfo.Branch != "" {
		b := arcInfo.Branch
		if arcInfo.Hash != "" {
			l := math2.MinInt(7, len(arcInfo.Hash))
			b = fmt.Sprintf("%s:%s", b, arcInfo.Hash[:l])
		}
		revMsg = fmt.Sprintf("arc branch %s", b)
	}
	if revMsg == "" {
		return ""
	}
	if arcStatus, err := arcutil.ArcStatus(); err == nil {
		s := arcStatus.Status
		isDirty := len(s.Changed) != 0 || len(s.Untracked) != 0 || len(s.Staged) != 0
		if isDirty {
			revMsg = fmt.Sprintf("%s (dirty checkout)", revMsg)
		}
	}
	return revMsg
}

func makeDefaultRevisionComment() string {
	msg := "infractl put"
	user := os.Getenv("USER")
	if user == "" {
		user = os.Getenv("USERNAME")
	}
	if user != "" {
		msg = fmt.Sprintf("%s by %s", msg, user)
	}
	hostname := os.Getenv("HOSTNAME")
	if hostname != "" {
		msg = fmt.Sprintf("%s at %s", msg, hostname)
	}
	revMsg := makeArcRevisionMsg()
	if revMsg == "" {
		return msg
	}
	return fmt.Sprintf("%s, %s", msg, revMsg)
}

func Put() *cobra.Command {
	cmd := &cobra.Command{
		Use:   "put {file.yaml}...",
		Short: "Put yaml specs into k8s",
		Args:  cobra.MatchAll(cobra.MinimumNArgs(1), root.ValidFile),
		Run: func(cmd *cobra.Command, args []string) {
			if !waitDeployed && waitDeployedTimeout != 0 {
				log.Fatalf("--timeout can be used only with --wait flag")
			}

			specs := []runtime.Object{}
			substs := substitutio.Load(substitutionsPath)
			for _, arg := range args {
				fileSpecs, err := serialization.ParseYaml(arg)
				if err != nil {
					// TODO (torkve) parser here just ignores unknown fields for concrete Kind, can we fix this behavior?
					log.Fatalf("Failed to parse spec %q: %v", arg, err)
				}
				if len(fileSpecs) == 0 {
					log.Fatalf("File %q contains no known objects", arg)
				}
				for _, fileSpec := range fileSpecs {
					err = substitutio.AugmentSpec(fileSpec, substs)
					if err != nil {
						log.Fatalf("Failed to substitute %q spec: %v", arg, err)
					}
					err = fillDefaultValues(fileSpec)
					if err != nil {
						log.Fatalf("Failed to detect %q default values: %v", arg, err)
					}
				}
				specs = append(specs, fileSpecs...)
			}

			sort.Slice(specs, func(i, j int) bool {
				kind1 := specs[i].GetObjectKind().GroupVersionKind().Kind
				kind2 := specs[j].GetObjectKind().GroupVersionKind().Kind

				// We reverse order here (> instead of <) so that any missing kinds
				// would get value 0 and be processed last
				return kindsOrder[kind1] > kindsOrder[kind2]
			})

			c := kubeutil.MakeClient()

			var waitErr error
			for _, spec := range specs {
				var err error
				var namespacedName string
				var changed bool
				kind := spec.GetObjectKind().GroupVersionKind().Kind
				switch kind {
				case "Namespace":
					if waitDeployed {
						log.Fatalf("--wait can be used only for DeployStage object kind")
					}
					obj := spec.(*v1.Namespace)
					namespacedName = obj.Name
					changed, err = c.PutObject(root.Context, client.ObjectKeyFromObject(obj), obj)
				case "AwacsUpstream":
					if waitDeployed {
						log.Fatalf("--wait can be used only for DeployStage object kind")
					}
					obj := spec.(*auv1.AwacsUpstream)
					namespacedName = obj.Namespace + "/" + obj.Name
					changed, err = c.PutObject(root.Context, client.ObjectKeyFromObject(obj), obj)
				case "AwacsBackend":
					if waitDeployed {
						log.Fatalf("--wait can be used only for DeployStage object kind")
					}
					obj := spec.(*abv1.AwacsBackend)
					namespacedName = obj.Namespace + "/" + obj.Name
					changed, err = c.PutObject(root.Context, client.ObjectKeyFromObject(obj), obj)
				case "Runtime":
					if waitDeployed {
						log.Fatalf("--wait can be used only for DeployStage object kind")
					}
					obj := spec.(*rv1.Runtime)
					if shareSecrets {
						err = shareMissingSecrets(root.Context, c, obj)
						if err != nil {
							log.Fatal(err.Error())
						}
					}
					namespacedName = obj.Namespace + "/" + obj.Name
					changed, err = c.PutObject(root.Context, client.ObjectKeyFromObject(obj), obj)
				case "DeployStage":
					if revisionComment == "" {
						revisionComment = makeDefaultRevisionComment()
					}
					obj := spec.(*dv1.DeployStage)
					if shareSecrets {
						err = shareMissingSecrets(root.Context, c, obj)
						if err != nil {
							log.Fatal(err.Error())
						}
					}
					if obj.Annotations == nil {
						obj.Annotations = map[string]string{}
					}
					obj.Annotations[dv1.RevisionCommentAnnotation] = revisionComment
					namespacedName = obj.Namespace + "/" + obj.Name
					changed, err = c.PutObject(root.Context, client.ObjectKeyFromObject(obj), obj)
					if err == nil && waitDeployed {
						stageChecker := deployedcheckers.NewStageDeployedChecker(
							types.NamespacedName{Namespace: obj.Namespace, Name: obj.Name},
							obj.Generation,
						)
						waitErr = conditionchecker.Wait(stageChecker, time.Duration(waitDeployedTimeout)*time.Second)
					}
				}
				if err != nil {
					log.Fatalf("Failed to put %v %q: %v", kind, namespacedName, err)
				} else {
					if waitDeployed {
						if waitErr != nil {
							log.Fatalf("Failed to wait %v %q deployed: %v\n", kind, namespacedName, waitErr)
						} else {
							if changed {
								log.Printf("%v %q deployed successfully\n", kind, namespacedName)
							} else {
								log.Printf("%v %q deployed successfully (unchanged)\n", kind, namespacedName)
							}
						}
					} else {
						if changed {
							log.Printf("Put %v %q\n", kind, namespacedName)
						} else {
							log.Printf("Put %v %q (unchanged)\n", kind, namespacedName)
						}
					}
				}
			}
		},
	}
	cmd.Flags().BoolVar(&shareSecrets, "share-secrets", shareSecrets, "automatically share secrets with deploying robot")
	cmd.Flags().StringVar(&substitutionsPath, "build-file", ".build.yaml", "path to build artifacts spec")
	cmd.Flags().StringVarP(&revisionComment, "message", "m", "", "human readable revision description. Applied only for DeployStage objects")
	cmd.Flags().BoolVarP(&waitDeployed, "wait", "w", false, "wait until YP stage is deployed")
	cmd.Flags().IntVarP(&waitDeployedTimeout, "timeout", "t", defaultWaitTimeout, "timeout for waiting until YP stage is deployed. By default blocks inifinetly")
	return cmd
}
