package put

import (
	"context"
	"fmt"
	"log"

	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/types"
	"k8s.io/apimachinery/pkg/util/sets"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/internal/secrets"
	oauthinfoclient "a.yandex-team.ru/infra/infractl/services/oauth-to-login/client"
	"a.yandex-team.ru/infra/infractl/util/kubeutil"
	"a.yandex-team.ru/infra/infractl/util/oauthutil"
	"a.yandex-team.ru/library/go/yandex/yav"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
)

type SecretContainer interface {
	client.Object
	CollectSecrets() sets.String
}

type UserSecretReader struct {
	c             kubeutil.Client
	yavOauthToken string
	ctx           context.Context
}

func (r *UserSecretReader) GetK8sClient() client.Client {
	return r.c
}

func (r *UserSecretReader) GetYavClient() *httpyav.Client {
	token := r.yavOauthToken
	if len(token) == 0 {
		token = oauthutil.MustGetToken(r.ctx, oauthutil.InfractlTokenEnvVarName)
		r.yavOauthToken = token
	}
	c, err := httpyav.NewClient(httpyav.WithOAuthToken(token))
	if err != nil {
		log.Fatalf("Failed to create Yav client: %v", err)
	}
	return c
}

func shareMissingSecrets(ctx context.Context, c kubeutil.Client, object SecretContainer) error {
	objSecrets := object.CollectSecrets()

	secretReader := &UserSecretReader{c: c, ctx: ctx}

	ns := &corev1.Namespace{}
	if err := client.IgnoreNotFound(c.Get(ctx, types.NamespacedName{Name: object.GetNamespace()}, ns)); err != nil {
		return fmt.Errorf("failed to get namespace %s: %w", object.GetNamespace(), err)
	}

	// To check if secret shared with robot on behalf of which infractl will manage
	// user's objects, first we need to determine this robot name.
	// The only way to do it is to read robot token which user provided to us earlier
	// and then using this token, obtain robot login from oauth
	token, err := secrets.LoadProviderTokenAsUser(secretReader, ctx, ns, "yp")
	if err != nil {
		return fmt.Errorf("failed to get YP token from namespace: %w", err)
	}

	oauthClient := oauthinfoclient.NewProductionClient()
	loginInfo, err := oauthClient.GetInfo(ctx, token)
	if err != nil {
		return fmt.Errorf("failed to resolve resolve token into robot login to share token with: %w", err)
	}

	var notDelegatedSecrets []string

	yavClient := secretReader.GetYavClient()

	for s := range objSecrets {
		delegated, err := secrets.CheckSecretReadable(ctx, yavClient, loginInfo.UID, s)
		if err != nil {
			return fmt.Errorf("failed to check secret %q: %w", s, err)
		}
		if !delegated {
			notDelegatedSecrets = append(notDelegatedSecrets, s)
		}
	}

	for _, s := range notDelegatedSecrets {
		if err = shareYavSecret(ctx, yavClient, s, loginInfo.UID); err != nil {
			return fmt.Errorf("failed to share secret %q: %w", s, err)
		} else {
			log.Printf("Secret %q has been shared with user %q", s, loginInfo.Login)
		}
	}
	return nil
}

func shareYavSecret(ctx context.Context, client *httpyav.Client, secretUUID string, robotUID uint64) error {
	req := yav.SecretRoleRequest{
		Role: "READER",
		UID:  robotUID,
	}
	rsp, err := client.AddSecretRole(ctx, secretUUID, req)
	if err != nil {
		return err
	}
	return rsp.Err()
}
