package arcutil

import (
	"encoding/json"
	"fmt"
	"os"
	"os/exec"
	"path/filepath"
	"runtime"
	"sync"

	"a.yandex-team.ru/library/go/yatool"
)

const windowsOS = "windows"

var (
	arcadiaRoot     string
	arcadiaRootErr  error
	arcadiaRootOnce sync.Once

	ya     string
	yaErr  error
	yaOnce sync.Once
)

// ArcadiaRoot returns the current Arcadia root
func ArcadiaRoot(arcPath string) (string, error) {
	arcadiaRootOnce.Do(func() {
		arcadiaRoot, arcadiaRootErr = yatool.FindArcadiaRoot(arcPath)
	})

	return arcadiaRoot, arcadiaRootErr
}

func Ya(arcPath string) (string, error) {
	yaOnce.Do(func() {
		var arcadiaRoot string
		arcadiaRoot, yaErr = ArcadiaRoot(arcPath)
		if yaErr != nil {
			return
		}

		ya, yaErr = yaForPath(arcadiaRoot)
	})

	return ya, yaErr
}

func ArcadiaRootAndYa(arcPath string) (root string, ya string, err error) {
	root, err = ArcadiaRoot(arcPath)
	if err != nil {
		return
	}
	ya, err = Ya(arcPath)
	return
}

func yaForPath(path string) (string, error) {
	yaBinary := "ya"
	if runtime.GOOS == windowsOS {
		yaBinary += ".bat"
	}

	yaPath := filepath.Join(path, yaBinary)
	if _, err := os.Stat(yaPath); err != nil {
		return "", err
	}

	return yaPath, nil
}

type arcInfo struct {
	Revision int    `json:"revision"`
	Branch   string `json:"branch"`
	Hash     string `json:"hash"`
}

func ArcInfo() (*arcInfo, error) {
	cmd := exec.Command("arc", "info", "--json")
	stdout, err := cmd.Output()
	if err != nil {
		return nil, fmt.Errorf("failed to execute `arc info --json`: %w", err)
	}
	rv := &arcInfo{}
	if err = json.Unmarshal(stdout, rv); err != nil {
		return nil, fmt.Errorf("failed to unmarshal `arc info --json` output: %w", err)
	}
	return rv, nil
}

type fileStatus struct {
	Path   string `json:"path"`
	Status string `json:"status"`
}

type arcStatus struct {
	Status struct {
		Changed   []*fileStatus `json:"changed,omitempty"`
		Untracked []*fileStatus `json:"untracked,omitempty"`
		Staged    []*fileStatus `json:"staged,omitempty"`
	} `json:"status"`
}

func ArcStatus() (*arcStatus, error) {
	cmd := exec.Command("arc", "st", "--json")
	stdout, err := cmd.Output()
	if err != nil {
		return nil, fmt.Errorf("failed to execute `arc st --json`: %w", err)
	}
	s := &arcStatus{}
	if err = json.Unmarshal(stdout, s); err != nil {
		return nil, fmt.Errorf("failed to unmarshal `arc st --json` output: %w", err)
	}
	return s, nil
}
