package ayaml

import (
	"bytes"
	"encoding/json"
	"path/filepath"
	"text/template"

	"github.com/ghodss/yaml"

	rtv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
)

const tmplString string = `
service: {{.ABC}}
title: Deploy {{.Service}}
ci:
  release-title-source: flow
  secret: {{.SecretUUID}}
  runtime:
    sandbox-owner: {{.SandboxGroup}}
  releases:
    autotest:
      stages:
        - id: start
          title: Start
        - id: build
          title: Build {{.Service}}
          displace: true
        - id: release-to-prestable
          title: Release to prestable
          displace: true
        - id: release-to-stable
          title: Release to stable
      filters:
      - abs-paths:
        - {{.AbsPath}}
  flows:
    autobuild:
      title: Deploy {{.Service}}
      jobs:
        start:
          task: dummy
          stage: start
        build:
          task: common/arcadia/ya_package_2
          title: Build
          stage: build
          needs: start
          input: {{.Build}}
        release-to-prestable:
          task: common/infractl/apply_specs
          title: Release pre-{{.Service}}
          needs: build
          stage: release-to-prestable
          input:
            config:
              source:
                arc:
                  path: {{.SpecPathPrestable}}
        release-to-stable:
          task: common/infractl/apply_specs
          title: Release prod-{{.Service}}
          needs: release-to-prestable
          stage: release-to-stable
          input:
            config:
              source:
                arc:
                  path: {{.SpecPathStable}}
          manual:
            enabled: true
            prompt: "Approve deploy to stable?"
`

type GenerateOptions struct {
	SandboxGroup      string
	SecretUUID        string
	SpecPathPrestable string
	SpecPathStable    string
}

type Generator interface {
	Generate(rt *rtv1.Runtime, opts GenerateOptions) ([]byte, error)
}

type BaseGenerator struct {
	ABCSlug       string
	BuildFilePath string
}

func (g *BaseGenerator) generateAYAML(service string, build map[string]interface{}, opts GenerateOptions) ([]byte, error) {
	tmpl, err := template.New("a.yaml").Parse(tmplString)
	if err != nil {
		return nil, err
	}
	// We cannot inline yaml into yaml, because yaml is indent respectful format. So we convert build map to json and
	// inline json into yaml (every json string is a valid yaml string).
	jsonBuild, err := json.Marshal(build)
	if err != nil {
		return nil, err
	}

	var rendered bytes.Buffer
	data := struct {
		ABC               string
		Service           string
		SecretUUID        string
		SandboxGroup      string
		SpecPathPrestable string
		SpecPathStable    string
		Build             string
		AbsPath           string
	}{
		ABC:               g.ABCSlug,
		Service:           service,
		SecretUUID:        opts.SecretUUID,
		SandboxGroup:      opts.SandboxGroup,
		SpecPathPrestable: opts.SpecPathPrestable,
		SpecPathStable:    opts.SpecPathStable,
		Build:             string(jsonBuild),
		AbsPath:           filepath.Join(filepath.Dir(g.BuildFilePath), "**"),
	}
	if err = tmpl.Execute(&rendered, data); err != nil {
		return nil, err
	}
	// For now rendered yaml has build section in json format and it is not pretty. To convert it to yaml
	// just marshal and unmarshal full rendered yaml.
	unmarshalled := new(map[string]interface{})
	if err = yaml.Unmarshal(rendered.Bytes(), unmarshalled); err != nil {
		return nil, err
	}
	renderedYAML, err := yaml.Marshal(unmarshalled)
	if err != nil {
		return nil, err
	}
	return renderedYAML, nil
}

type SandboxGenerator struct {
	BaseGenerator
}

func (g *SandboxGenerator) Generate(rt *rtv1.Runtime, opts GenerateOptions) ([]byte, error) {
	build := map[string]interface{}{
		"packages":      g.BuildFilePath,
		"package_type":  "tarball",
		"resource_type": "TEST_RESOURCE",
	}
	return g.BaseGenerator.generateAYAML(rt.GetName(), build, opts)
}

type DockerGenerator struct {
	BaseGenerator
	ImageRepository string
	User            string
	DockerYAVToken  string
}

func (g *DockerGenerator) Generate(rt *rtv1.Runtime, opts GenerateOptions) ([]byte, error) {
	build := map[string]interface{}{
		"packages":                g.BuildFilePath,
		"package_type":            "docker",
		"docker_image_repository": g.ImageRepository,
		"docker_user":             g.User,
		"docker_yav_token":        g.DockerYAVToken,
		"docker_push_image":       true,
		"custom_version":          "${context.version_info.full}",
	}
	return g.BaseGenerator.generateAYAML(rt.GetName(), build, opts)
}
