package deployedcheckers

import (
	"fmt"

	"k8s.io/apimachinery/pkg/types"

	"a.yandex-team.ru/infra/infractl/cli/commands/root"
	deployinfrav1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	"a.yandex-team.ru/infra/infractl/util/kubeutil"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

// StageDeployedChecker implements conditionchecker.ConditionChecker interface
type StageDeployedChecker struct {
	stageName               types.NamespacedName
	generationCheckDeployed int64
	kubeclient              kubeutil.Client
}

func NewStageDeployedChecker(stageName types.NamespacedName, generationCheckDeployed int64) *StageDeployedChecker {
	return &StageDeployedChecker{
		stageName:               stageName,
		generationCheckDeployed: generationCheckDeployed,
		kubeclient:              kubeutil.MakeClient(),
	}
}

func (c *StageDeployedChecker) makeReasonMessage(msg string) string {
	return fmt.Sprintf("DeployStage \"%s\": %s", c.stageName, msg)
}

func (c *StageDeployedChecker) makeError(msg string) error {
	return fmt.Errorf("DeployStage \"%s\": %s", c.stageName, msg)
}

func (c *StageDeployedChecker) addToInProgress(inProgress *[]string, name string, readyStatus *ypapi.TCondition, appliedRev int, currentRev int) error {
	if appliedRev < currentRev {
		return c.makeError("OUTDATED: Stage spec was updated in YP directly, bypassing infractl")
	}
	if appliedRev > currentRev || readyStatus.GetStatus() != ypapi.EConditionStatus_CS_TRUE {
		*inProgress = append(*inProgress, name)
	}
	return nil
}

func (c *StageDeployedChecker) ConditionMet() (bool, string, error) {
	stage := &deployinfrav1.DeployStage{}
	if err := c.kubeclient.Get(root.Context, c.stageName, stage); err != nil {
		// we don't return error here, because we don't want to stop waiting deployed if we couldn't fetch stage.
		// Just retry to fetch on the next iteration
		return false, c.makeReasonMessage(err.Error()), nil
	}
	if stage.Generation > c.generationCheckDeployed {
		return false, "", c.makeError("CONFLICT: spec was updated concurrently")
	}

	generation := stage.Status.GetSyncStatus().GetAppliedGeneration()
	if generation < stage.GetGeneration() {
		return false, c.makeReasonMessage("UPDATING YP SPEC: controller hasn't updated spec in YP yet..."), nil
	}

	inProgressDUs := make([]string, 0, len(stage.Status.GetStageStatus().GetDeployUnits()))
	ypRevs := stage.Status.GetAppliedDeployRevisions()
	for duName, du := range stage.Status.GetStageStatus().GetDeployUnits() {
		appliedRev := ypRevs.DeployUnits[duName]
		err := c.addToInProgress(
			&inProgressDUs,
			duName,
			du.GetReady(),
			int(appliedRev),
			int(du.GetTargetRevision()))
		if err != nil {
			return false, "", err
		}
	}

	inProgressDRs := make([]string, 0, len(stage.Status.GetStageStatus().GetDynamicResources()))
	for drName, dr := range stage.Status.GetStageStatus().GetDynamicResources() {
		appliedRev := ypRevs.DynamicResources[drName]
		err := c.addToInProgress(
			&inProgressDRs,
			drName,
			dr.GetStatus().GetReady().GetCondition(),
			int(appliedRev),
			int(dr.GetStatus().GetRevision()))
		if err != nil {
			return false, "", err
		}
	}

	if len(inProgressDUs) == 0 && len(inProgressDRs) == 0 {
		return true, c.makeReasonMessage("DEPLOY FINISHED"), nil
	}
	return false, c.makeReasonMessage("DEPLOYING..."), nil
}
