package iyaml

import (
	"fmt"
	"path"

	"github.com/magiconair/properties"
	"google.golang.org/protobuf/proto"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/cli/commands/make/manifest"
	"a.yandex-team.ru/infra/infractl/cli/internal/deploystage"
	dsv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	rv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	deployclient "a.yandex-team.ru/infra/infractl/internal/deploy/client"
	"a.yandex-team.ru/infra/infractl/util/deployutil"
	"a.yandex-team.ru/yp/go/proto/podagent"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

type EnvPatcher struct {
	raw          map[string]string
	source       string
	selector     *manifest.ContainerSelector
	manifestRoot string
}

func NewEnvPatcher(
	raw map[string]string,
	source string,
	selector *manifest.ContainerSelector,
	manifestRoot string,
) *EnvPatcher {
	return &EnvPatcher{
		raw:          raw,
		source:       source,
		selector:     selector,
		manifestRoot: manifestRoot,
	}
}

func (p *EnvPatcher) collectEnvVars() (map[string]string, error) {
	vars := map[string]string{}

	for key, value := range p.raw {
		vars[key] = value
	}

	if len(p.source) == 0 {
		return vars, nil
	}

	props, err := properties.LoadFile(path.Join(p.manifestRoot, p.source), properties.UTF8)
	if err != nil {
		return nil, fmt.Errorf("env var source file %s read failed: %w", p.source, err)
	}
	for key, value := range props.Map() {
		if _, ok := vars[key]; ok {
			return nil, fmt.Errorf(
				"invalid env source file %s: duplicate env var name: %s",
				p.source, key,
			)
		}
		vars[key] = value
	}

	return vars, nil
}

func (p *EnvPatcher) getDeployStageEnvs(du *ypapi.TDeployUnitSpec) (*[]*podagent.TEnvVar, error) {
	_, box, err := deploystage.DeduceBox(du, p.selector.GetBoxId())
	if err != nil {
		return nil, err
	}

	env := &box.Env
	if workloadID := p.selector.GetWorkloadId(); len(workloadID) != 0 {
		_, workload, err := deploystage.DeduceWorkload(du, workloadID, box.Id)
		if err != nil {
			return nil, err
		}
		env = &workload.Env
	}
	return env, nil
}

func validateNewDeployStageEnvs(existingVars []*podagent.TEnvVar, newVars map[string]string) error {
	existingKeySet := make(map[string]bool, len(existingVars))
	for _, envVar := range existingVars {
		existingKeySet[envVar.Name] = true
	}

	for key := range newVars {
		if _, ok := existingKeySet[key]; ok {
			return fmt.Errorf("duplicate env var name: %s", key)
		}
	}

	return nil
}

func (p *EnvPatcher) patchDeployStage(s *dsv1.DeployStage, vars map[string]string) error {
	_, du, err := deploystage.DeduceDeployUnit(s.Spec.StageSpec, p.selector.GetDeployUnitId())
	if err != nil {
		return err
	}

	env, err := p.getDeployStageEnvs(du)
	if err != nil {
		return err
	}
	if err := validateNewDeployStageEnvs(*env, vars); err != nil {
		return err
	}

	podAgentEnvVars, newSecretRefs := deployutil.ParseEnvValues(vars, true)

	*env = append(*env, podAgentEnvVars...)

	podSpec := deployclient.GеtPodSpec(du)
	if podSpec.SecretRefs == nil {
		podSpec.SecretRefs = make(map[string]*ypapi.TSecretRef, len(newSecretRefs))
	}
	for alias, value := range newSecretRefs {
		if oldValue, ok := podSpec.SecretRefs[alias]; ok {
			if !proto.Equal(oldValue, value) {
				return fmt.Errorf("stage already has secret ref %s with different content", alias)
			}
		}
		podSpec.SecretRefs[alias] = value
	}

	return nil
}

func (p *EnvPatcher) patchRuntime(r *rv1.Runtime, vars map[string]string) error {
	env := &r.Spec.Env
	if workflowID := p.selector.GetWorkloadId(); len(workflowID) != 0 {
		workflow, ok := r.Spec.Workloads[workflowID]
		if !ok {
			return fmt.Errorf("workflow %s not found", workflowID)
		}
		env = &workflow.Env
	}

	if *env == nil {
		*env = make(map[string]string, len(vars))
	}

	for key, value := range vars {
		if _, ok := (*env)[key]; ok {
			return fmt.Errorf("duplicate key: %s", key)
		}
		(*env)[key] = value
	}

	return nil
}

func (p *EnvPatcher) Patch(objects []client.Object) ([]client.Object, error) {
	name := p.selector.GetName()

	vars, err := p.collectEnvVars()
	if err != nil {
		return nil, err
	}

	objectsPatched := 0
	for _, object := range objects {
		stage, stageOK := object.(*dsv1.DeployStage)
		runtime, runtimeOK := object.(*rv1.Runtime)
		if !stageOK && !runtimeOK {
			continue
		}

		if name != "" && object.GetName() != name {
			continue
		}

		var err error
		switch {
		case stageOK:
			err = p.patchDeployStage(stage, vars)
		case runtimeOK:
			err = p.patchRuntime(runtime, vars)
		}
		if err != nil {
			return nil, fmt.Errorf("cannot patch %s: %w", object.GetName(), err)
		}
		objectsPatched++
	}

	if objectsPatched == 0 {
		return nil, fmt.Errorf("selector found neither DeployStage nor Runtime")
	}

	return objects, nil
}
