package iyaml

import (
	"fmt"
	"io/ioutil"
	"os"
	"path"
	"sort"
	"strings"

	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/cli/commands/make/manifest"
	"a.yandex-team.ru/infra/infractl/cli/internal/deploystage"
	dv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	rv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	"a.yandex-team.ru/yp/go/proto/podagent"
)

type InsertFilesPatcher struct {
	files        map[string]string
	dirs         map[string]string
	selector     *manifest.ContainerSelector
	manifestRoot string
}

func NewInsertFilesPatcher(
	files map[string]string,
	dirs map[string]string,
	selector *manifest.ContainerSelector,
	manifestRoot string,
) *InsertFilesPatcher {
	return &InsertFilesPatcher{
		files:        files,
		dirs:         dirs,
		selector:     selector,
		manifestRoot: manifestRoot,
	}
}

type file struct {
	dir     string
	name    string
	content string
}

type resource struct {
	mountPoint string
	files      []file
}

func (p *InsertFilesPatcher) readFiles(inputFiles map[string]string) ([]file, error) {
	files := []file{}

	for boxFile, patchFile := range inputFiles {
		content, err := ioutil.ReadFile(path.Join(p.manifestRoot, patchFile))
		if err != nil {
			return nil, fmt.Errorf("unable to read %s: %w", patchFile, err)
		}
		dir, name := path.Split(boxFile)
		files = append(files, file{
			dir:     dir,
			name:    name,
			content: string(content),
		})
	}

	return files, nil
}

func (p *InsertFilesPatcher) readDirs() ([]file, error) {
	var files []file

	for dir, localDir := range p.dirs {
		dirEntries, err := os.ReadDir(path.Join(p.manifestRoot, localDir))
		if err != nil {
			return nil, fmt.Errorf("unable to read directory %q: %w", localDir, err)
		}
		dirFiles := map[string]string{}
		for _, dirEntry := range dirEntries {
			if dirEntry.IsDir() {
				return nil, fmt.Errorf(
					"directory %s cannot be used as a source of files to be mounted because it contains subdirectories", dir,
				)
			}
			name := dirEntry.Name()
			dirFiles[path.Join(dir, name)] = path.Join(localDir, name)
		}
		newFiles, err := p.readFiles(dirFiles)
		if err != nil {
			return nil, err
		}
		files = append(files, newFiles...)
	}

	return files, nil
}

func (p *InsertFilesPatcher) groupFilesByMountPoint() ([]*resource, error) {
	mountPoints := map[string]*resource{}
	// slice to maintain stable resources order
	var resources []*resource

	files, err := p.readFiles(p.files)
	if err != nil {
		return nil, err
	}
	filesFromDirs, err := p.readDirs()
	if err != nil {
		return nil, err
	}
	files = append(files, filesFromDirs...)

	sort.Slice(files, func(i, j int) bool {
		if files[i].dir != files[j].dir {
			return files[i].dir < files[j].dir
		}
		return files[i].name < files[j].name
	})

	for _, f := range files {
		r, ok := mountPoints[f.dir]
		if !ok {
			r = &resource{mountPoint: f.dir}
			resources = append(resources, r)
			mountPoints[f.dir] = r
		}
		r.files = append(r.files, f)
	}

	return resources, nil
}

func (p *InsertFilesPatcher) patchDeployStage(s *dv1.DeployStage, resources []*resource) error {
	_, du, err := deploystage.DeduceDeployUnit(s.Spec.StageSpec, p.selector.GetDeployUnitId())
	if err != nil {
		return err
	}
	_, box, err := deploystage.DeduceBox(du, p.selector.GetBoxId())
	if err != nil {
		return err
	}

	existingResources := deploystage.GetPodAgentSpec(du).Resources
	resourcesMap := map[string]*podagent.TResource{}
	for _, resource := range existingResources.StaticResources {
		resourcesMap[resource.Id] = resource
	}

	mountedResourcesMap := map[string]*podagent.TMountedStaticResource{}
	for _, resource := range box.StaticResources {
		mountedResourcesMap[resource.MountPoint] = resource
	}

	for _, r := range resources {
		podagentFiles := make([]*podagent.TFile, 0, len(r.files))
		for _, f := range r.files {
			podagentFiles = append(podagentFiles, &podagent.TFile{
				FileName: f.name,
				Content: &podagent.TFile_RawData{
					RawData: string(f.content),
				},
			})
		}

		// add files to existing resource if possible
		if mountedResource, ok := mountedResourcesMap[r.mountPoint]; ok {
			podAgentResouce := resourcesMap[mountedResource.ResourceRef]
			switch podAgentResouce.DownloadMethod.(type) {
			case *podagent.TResource_Files:
				files := podAgentResouce.GetFiles()
				files.Files = append(files.Files, podagentFiles...)
			default:
				return fmt.Errorf(
					"cannot mount %s, resource of type other than 'Files' mounted there",
					r.mountPoint,
				)
			}
			continue
		}

		resourceID := strings.ReplaceAll(strings.Trim(r.mountPoint, "/"), "/", "-")
		podAgentResource := &podagent.TResource{
			Id: resourceID,
			DownloadMethod: &podagent.TResource_Files{
				Files: &podagent.TFiles{
					Files: podagentFiles,
				},
			},
			Verification: &podagent.TVerification{
				Checksum: "EMPTY:",
			},
		}
		existingResources.StaticResources = append(existingResources.StaticResources, podAgentResource)
		box.StaticResources = append(box.StaticResources, &podagent.TMountedStaticResource{
			ResourceRef: resourceID,
			MountPoint:  r.mountPoint,
		})
	}

	return nil
}

func (p *InsertFilesPatcher) patchRuntime(runtime *rv1.Runtime, resources []*resource) error {
	if runtime.Spec.GetFiles() == nil {
		runtime.Spec.Files = make(map[string]string, len(resources))
	}
	for _, r := range resources {
		for _, file := range r.files {
			fileKey := path.Join(file.dir, file.name)
			if _, ok := runtime.Spec.Files[fileKey]; ok {
				return fmt.Errorf("file %s already exists", fileKey)
			}
			runtime.Spec.Files[fileKey] = file.content
		}
	}

	return nil
}

func (p *InsertFilesPatcher) Patch(objects []client.Object) ([]client.Object, error) {
	name := p.selector.GetName()

	files, err := p.groupFilesByMountPoint()
	if err != nil {
		return nil, err
	}

	objectsPatched := 0
	for _, object := range objects {
		stage, stageOK := object.(*dv1.DeployStage)
		runtime, runtimeOK := object.(*rv1.Runtime)
		if !stageOK && !runtimeOK {
			continue
		}

		if name != "" && object.GetName() != name {
			continue
		}

		var err error
		switch {
		case stageOK:
			err = p.patchDeployStage(stage, files)
		case runtimeOK:
			err = p.patchRuntime(runtime, files)
		}
		if err != nil {
			return nil, fmt.Errorf("cannot patch %s: %w", object.GetName(), err)
		}
		objectsPatched++
	}

	if objectsPatched == 0 {
		return nil, fmt.Errorf("selector found neither DeployStage nor Runtime")
	}

	return objects, nil
}
