package iyaml

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"os/exec"
	"path"

	"github.com/ghodss/yaml"
	"google.golang.org/protobuf/types/known/structpb"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/cli/internal/arcutil"
	"a.yandex-team.ru/infra/infractl/cli/internal/serialization"
)

const (
	OpenAPISchemaPath     = "infra/infractl/kustomize/bases/schemas.json"
	KustomizationFileName = "kustomization.yaml"
	ResourcesFileName     = "resources.yaml"
)

type Patch struct {
	Path string `json:"path,omitempty" yaml:"path,omitempty"`
}

type Kustomization struct {
	OpenAPI               map[string]string `json:"openapi,omitempty" yaml:"openapi,omitempty"`
	PatchesStrategicMerge []string          `json:"patchesStrategicMerge,omitempty" yaml:"patchesStrategicMerge,omitempty"`
	PatchesJSON6902       []Patch           `json:"patchesJson6902,omitempty" yaml:"patchesJson6902,omitempty"`
	Patches               []Patch           `json:"patches,omitempty" yaml:"patches,omitempty"`
	Resources             []string          `json:"resources,omitempty" yaml:"resources,omitempty"`
	Crds                  []string          `json:"crds,omitempty" yaml:"crds,omitempty"`
}

func copyFile(src string, dst string) error {
	path, _ := path.Split(dst)
	if err := os.MkdirAll(path, 0755); err != nil {
		return fmt.Errorf("unable to create directory %s: %w", path, err)
	}

	srcFile, err := os.Open(src)
	if err != nil {
		return fmt.Errorf("unable to open file %s: %w", src, err)
	}
	dstFile, err := os.Create(dst)
	if err != nil {
		return fmt.Errorf("unable to open file %s: %w", dst, err)
	}

	if _, err := io.Copy(dstFile, srcFile); err != nil {
		return fmt.Errorf("unable to copy from %s to %s: %w", src, dst, err)
	}
	return nil
}

type KustomizePatcher struct {
	patch             *structpb.Struct
	kustomizationRoot string
	arcRoot           string
}

func NewKustomizePatcher(
	patch *structpb.Struct,
	kustomizationRoot, arcRoot string,
) *KustomizePatcher {
	return &KustomizePatcher{
		patch:             patch,
		kustomizationRoot: kustomizationRoot,
		arcRoot:           arcRoot,
	}
}

func (p *KustomizePatcher) copyFilesToWorkDir(filePaths []string, workDir string) error {
	for _, filePath := range filePaths {
		pathSrc := path.Join(p.kustomizationRoot, filePath)
		pathDst := path.Join(workDir, filePath)
		if err := copyFile(pathSrc, pathDst); err != nil {
			return err
		}
	}
	return nil
}

func (p *KustomizePatcher) prepareWorkDir(objects []client.Object, workDir string) error {
	patchRaw, err := p.patch.MarshalJSON()
	if err != nil {
		return err
	}

	invalidKustomizeErr := fmt.Errorf("invalid kustomize patch spec: %w", err)
	kustomization := Kustomization{}
	if err := json.Unmarshal(patchRaw, &kustomization); err != nil {
		return invalidKustomizeErr
	}
	fullSpec := map[string]interface{}{}
	if err := json.Unmarshal(patchRaw, &fullSpec); err != nil {
		return invalidKustomizeErr
	}

	filesToCopy := []string{}
	// PatchesStrategicMerge may be inline
	// https://kubectl.docs.kubernetes.io/references/kustomize/kustomization/patchesstrategicmerge/
	for _, f := range kustomization.PatchesStrategicMerge {
		filePath := path.Join(p.kustomizationRoot, f)
		isFile, err := isFileExists(filePath)
		if err != nil {
			return err
		}
		if isFile {
			filesToCopy = append(filesToCopy, f)
		}
	}
	patchesWithPath := append(kustomization.PatchesJSON6902, kustomization.Patches...)
	for _, patch := range patchesWithPath {
		// patch may be inline
		if len(patch.Path) != 0 {
			filesToCopy = append(filesToCopy, patch.Path)
		}
	}
	filesToCopy = append(filesToCopy, kustomization.Crds...)

	if err := p.copyFilesToWorkDir(filesToCopy, workDir); err != nil {
		return err
	}

	_, openapiFileName := path.Split(OpenAPISchemaPath)
	err = copyFile(path.Join(p.arcRoot, OpenAPISchemaPath), path.Join(workDir, openapiFileName))
	if err != nil {
		return err
	}

	if err := serialization.DumpObjectsToFile(objects, path.Join(workDir, ResourcesFileName)); err != nil {
		return err
	}

	fullSpec["openapi"] = map[string]string{"path": openapiFileName}
	fullSpec["resources"] = []string{ResourcesFileName}
	kustomizationYAML, err := yaml.Marshal(fullSpec)
	if err != nil {
		return fmt.Errorf("cannot create intermediate kustomization.yaml to apply patch: %w", err)
	}

	return ioutil.WriteFile(path.Join(workDir, KustomizationFileName), kustomizationYAML, 0644)
}

func (p *KustomizePatcher) runKustomize(workDir string) ([]client.Object, error) {
	ya, err := arcutil.Ya(p.arcRoot)
	if err != nil {
		return nil, fmt.Errorf("unable to detect ya binary: %w", err)
	}

	cmd := exec.Command(ya, "tool", "kubectl", "kustomize", workDir)

	var stderr, stdout bytes.Buffer
	cmd.Stderr = &stderr
	cmd.Stdout = &stdout

	err = cmd.Run()
	if err != nil {
		return nil, fmt.Errorf(
			"kustomize execution failed with stderr:\n%s\n%w",
			stderr.Bytes(),
			err,
		)
	}

	newObjects, err := serialization.ParseReaderTyped(&stdout)
	if err != nil {
		return nil, fmt.Errorf("cannot parse objects from kustomize execution: %w", err)
	}

	return newObjects, nil
}

func (p *KustomizePatcher) Patch(objects []client.Object) ([]client.Object, error) {
	workDir, err := ioutil.TempDir("", "kustomize")
	if err != nil {
		return nil, fmt.Errorf("temp dir creation failed: %w", err)
	}
	defer os.RemoveAll(workDir)

	if err := p.prepareWorkDir(objects, workDir); err != nil {
		return nil, err
	}

	return p.runKustomize(workDir)
}
