package iyaml

import (
	"errors"
	"fmt"
	"io/fs"
	"io/ioutil"
	"os"
	"path"
	"regexp"
	"strings"

	"github.com/ghodss/yaml"
	"google.golang.org/protobuf/encoding/protojson"
	"google.golang.org/protobuf/types/known/structpb"

	"a.yandex-team.ru/infra/infractl/cli/commands/make/manifest"
	"a.yandex-team.ru/infra/infractl/util/kubeutil"
)

const (
	DefaultDiscoveryPolicy         = manifest.DiscoveryPolicy_all
	ManifestFileName               = "i.yaml"
	ArcadiaCIFileName              = "a.yaml"
	ObjectFileNamePrefix           = "i."
	ObjectFileNameRegexpPattern    = `^i\.([a-z]+)\.([a-z-]+)\.?(patch)?\.yaml$`
	PatchSign                      = "patch"
	APIVersion                     = "k.yandex-team.ru/v1"
	namePrefixFieldName            = "namePrefix"
	patchesStrategicMergeFiledName = "patchesStrategicMerge"
)

var (
	ObjectFileNameRegexp = regexp.MustCompile(ObjectFileNameRegexpPattern)
)

func fillKubeObjectRequiredFields(data *manifest.ResourceManifest, apiVersion, kind, metaName string) (err error) {
	if data.ApiVersion == "" {
		data.ApiVersion = apiVersion
	}
	if data.Kind == "" {
		data.Kind = kind
	}
	meta := data.GetMetadata()
	if meta == nil {
		meta, err = structpb.NewStruct(map[string]interface{}{"name": metaName})
		if err != nil {
			return err
		}
		data.Metadata = meta
	}
	_, ok := meta.GetFields()["name"]
	if !ok {
		metaNameValue := structpb.NewStringValue(metaName)
		meta.GetFields()["name"] = metaNameValue
	}
	return nil
}

type ResourceFile struct {
	Path string
	Data *manifest.ResourceManifest
}

func NewResourceFile(dir string, fileName string, objKind string, objName string, content []byte) (*ResourceFile, error) {
	filePath := path.Join(dir, fileName)

	jsonContent, err := yaml.YAMLToJSON(content)
	if err != nil {
		return nil, fmt.Errorf("cannot decode to json %s: %w", filePath, err)
	}
	p := &protojson.UnmarshalOptions{DiscardUnknown: true}

	data := &manifest.ResourceManifest{}
	if err := p.Unmarshal(jsonContent, data); err != nil {
		return nil, fmt.Errorf("incorrect format in %s: %w", filePath, err)
	}

	err = fillKubeObjectRequiredFields(data, APIVersion, objKind, objName)
	if err != nil {
		return nil, err
	}

	return &ResourceFile{
		Path: filePath,
		Data: data,
	}, nil
}

func (r *ResourceFile) GenerateResourceBaseInline() (string, error) {
	base := &manifest.ResourceManifest{
		ApiVersion: r.Data.ApiVersion,
		Kind:       r.Data.Kind,
		Metadata:   r.Data.Metadata,
		Spec:       r.Data.Spec,
	}
	jsonData, err := protojson.Marshal(base)
	if err != nil {
		return "", err
	}
	res, err := yaml.JSONToYAML(jsonData)
	if err != nil {
		return "", err
	}
	return string(res), nil
}

func (r *ResourceFile) GenerateKustomizePatch(namePrefix string) (*manifest.Patch, error) {
	kustomizeData := map[string]interface{}{
		namePrefixFieldName: namePrefix,
	}
	isSpecExists := r.Data.Spec != nil
	if isSpecExists {
		mergeString, err := r.GenerateResourceBaseInline()
		if err != nil {
			return nil, fmt.Errorf("GenerateKustomizeMergeString: %w", err)
		}
		kustomizeData[patchesStrategicMergeFiledName] = []interface{}{mergeString}
	}

	kustomiseStruct, err := structpb.NewStruct(kustomizeData)
	if err != nil {
		return nil, fmt.Errorf("NewStruct: %w", err)
	}

	return &manifest.Patch{
		Kustomize: kustomiseStruct,
	}, nil
}

func (r *ResourceFile) GeneratePatches(namePrefix string, generateKustomise bool) ([]*manifest.Patch, error) {
	patches := r.Data.GetPatches()
	if patches == nil {
		patches = []*manifest.Patch{}
	}
	if generateKustomise {
		kustomisePatch, err := r.GenerateKustomizePatch(namePrefix)
		if err != nil {
			return nil, fmt.Errorf("GenerateKustomizePatch: %w", err)
		}
		patches = append([]*manifest.Patch{kustomisePatch}, patches...)
	}

	return patches, nil
}

type collectedResources struct {
	resources            []*ResourceFile
	resourcesWithPatches []*ResourceFile
	patches              []*ResourceFile
}

func collectResources(client kubeutil.Client, dir string) (*collectedResources, error) {
	files, err := ioutil.ReadDir(dir)
	if err != nil {
		return nil, err
	}
	fileNames, err := filterFileNames(dir, files)
	if err != nil {
		return nil, err
	}
	resources := make([]*ResourceFile, 0, len(fileNames))
	resourcesWithPatches := make([]*ResourceFile, 0, len(fileNames))
	patches := make([]*ResourceFile, 0, len(fileNames))
	for _, fileName := range fileNames {
		matches := ObjectFileNameRegexp.FindStringSubmatch(fileName)
		gvk, err := client.ResolveKind(matches[1])
		if err != nil {
			return nil, fmt.Errorf("%s in file name id not valid kind: %w", matches[1], err)
		}
		objName := matches[2]
		isPatch := matches[3] == PatchSign
		filePath := path.Join(dir, fileName)
		content, err := os.ReadFile(filePath)
		if err != nil {
			return nil, err
		}
		resourceFile, err := NewResourceFile(dir, fileName, gvk.Kind, objName, content)
		if err != nil {
			return nil, err
		}

		if isPatch {
			patches = append(patches, resourceFile)
		} else if len(resourceFile.Data.GetPatches()) > 0 {
			resourcesWithPatches = append(resourcesWithPatches, resourceFile)
		} else {
			resources = append(resources, resourceFile)
		}
	}
	return &collectedResources{
		resources:            resources,
		resourcesWithPatches: resourcesWithPatches,
		patches:              patches,
	}, nil

}

func isFileExists(filePath string) (bool, error) {
	if _, err := os.Stat(filePath); err == nil {
		return true, nil
	} else if errors.Is(err, os.ErrNotExist) {
		return false, nil
	} else {
		return false, err
	}
}

func filterFileNames(dir string, files []fs.FileInfo) ([]string, error) {
	result := []string{}
	for _, f := range files {
		if f.Name() == ManifestFileName {
			continue
		}
		if !strings.HasPrefix(f.Name(), ObjectFileNamePrefix) {
			continue
		}
		if !ObjectFileNameRegexp.MatchString(f.Name()) {
			return nil, fmt.Errorf("%s does not matched pattern %s in dir %s",
				f, string(ObjectFileNameRegexpPattern), dir)
		}
		result = append(result, f.Name())
	}
	return result, nil
}

func IsDiscoveryNeeded(current manifest.DiscoveryPolicy, needed manifest.DiscoveryPolicy) bool {
	if current == manifest.DiscoveryPolicy_default {
		current = DefaultDiscoveryPolicy
	}

	if current == manifest.DiscoveryPolicy_all {
		return true
	}

	return current == needed
}

func IsDiscoveryBaseNeeded(manifestContent *manifest.Manifest, arcadiaCIFileExisits bool) bool {
	if !IsDiscoveryNeeded(manifestContent.Discovery, manifest.DiscoveryPolicy_base) {
		return false
	}
	if arcadiaCIFileExisits {
		return false
	}
	baseExists := false
	for _, r := range manifestContent.Resources {
		switch r.Path.(type) {
		case *manifest.Resource_Base:
			baseExists = true
		}
	}
	return !baseExists
}

func IsDiscoveryFilesNeeded(manifestContent *manifest.Manifest) bool {
	if !IsDiscoveryNeeded(manifestContent.Discovery, manifest.DiscoveryPolicy_files) {
		return false
	}
	filesExists := false
	for _, r := range manifestContent.Resources {
		switch r.Path.(type) {
		case *manifest.Resource_File:
			filesExists = true
		}
	}
	return !filesExists
}

func IsDiscoveryPatchesNeeded(manifestContent *manifest.Manifest) bool {
	if !IsDiscoveryNeeded(manifestContent.Discovery, manifest.DiscoveryPolicy_files) {
		return false
	}
	return len(manifestContent.Patches) == 0
}

func appendBase(dir string, arcRoot string, manifestContent *manifest.Manifest) {
	baseDir := path.Clean(path.Join(dir, ".."))
	basePath := strings.Replace(baseDir, arcRoot+"/", "", 1)
	manifestContent.Resources = append(manifestContent.Resources, &manifest.Resource{
		Path: &manifest.Resource_Base{
			Base: basePath,
		},
	})
}

func (p *manifestProcessor) autoFillManifestResources(files *collectedResources, manifestContent *manifest.Manifest) error {
	for _, objResource := range append(files.resources, files.resourcesWithPatches...) {
		resourceInline, err := objResource.GenerateResourceBaseInline()
		if err != nil {
			return err
		}
		manifestContent.Resources = append(manifestContent.Resources, &manifest.Resource{
			Path: &manifest.Resource_Inline{
				Inline: resourceInline,
			},
		})
	}
	return nil
}

func (p *manifestProcessor) autoFillManifestPatches(
	dir string,
	files *collectedResources,
	manifestContent *manifest.Manifest,
) error {
	namePrefix := manifestContent.NamePrefix
	if namePrefix == "" {
		namePrefix = fmt.Sprintf("%s-", path.Base(dir))
	}
	for _, objResource := range files.patches {
		patches, err := objResource.GeneratePatches(namePrefix, true)
		if err != nil {
			return fmt.Errorf("%s, GeneratePatches, %w", objResource.Path, err)
		}
		manifestContent.Patches = append(manifestContent.Patches, patches...)
	}
	for _, objResource := range files.resourcesWithPatches {
		patches, err := objResource.GeneratePatches(namePrefix, false)
		if err != nil {
			return fmt.Errorf("%s, GeneratePatches, %w", objResource.Path, err)
		}
		manifestContent.Patches = append(manifestContent.Patches, patches...)
	}
	return nil
}

func (p *manifestProcessor) inferManifest(dir string) (*manifest.Manifest, error) {
	ayamlPath := path.Join(dir, ArcadiaCIFileName)
	ayamlExists, err := isFileExists(ayamlPath)
	if err != nil {
		return nil, fmt.Errorf("cannot check if a.yaml file exists in dir %q: %w", dir, err)
	}
	ns := ""
	if ayamlExists {
		ns = path.Base(dir)
	}
	return &manifest.Manifest{Namespace: ns}, nil
}

func (p *manifestProcessor) loadOrInferManifest(dir string) (*manifest.Manifest, error) {
	manifestPath := path.Join(dir, ManifestFileName)
	buf, err := ioutil.ReadFile(manifestPath)
	if errors.Is(err, os.ErrNotExist) {
		return p.inferManifest(dir)
	} else if err != nil {
		return nil, fmt.Errorf("cannot read i.yaml in %s: %w", dir, err)
	}
	jsonContent, err := yaml.YAMLToJSON(buf)
	if err != nil {
		return nil, fmt.Errorf("cannot decode i.yaml to json in %s: %w", dir, err)
	}
	manifestContent := &manifest.Manifest{}
	if err := protojson.Unmarshal(jsonContent, manifestContent); err != nil {
		return nil, fmt.Errorf("incorrect format in %s: %w", manifestPath, err)
	}
	return manifestContent, nil
}

func (p *manifestProcessor) makeManifest(client kubeutil.Client, dir string) (*manifest.Manifest, error) {
	manifestContent, err := p.loadOrInferManifest(dir)
	if err != nil {
		return nil, err
	}
	if err = p.inferResourcesAndPatches(client, dir, p.arcRoot, manifestContent); err != nil {
		return nil, fmt.Errorf("cannot infer resources and patches in dir %q: %w", dir, err)
	}
	return manifestContent, nil
}
