package iyaml

import (
	"strings"
	"testing"

	"github.com/ghodss/yaml"
	"github.com/stretchr/testify/assert"
	"google.golang.org/protobuf/encoding/protojson"

	"a.yandex-team.ru/infra/infractl/cli/commands/make/manifest"
)

func makeValidYamlContent(s string) []byte {
	s = strings.ReplaceAll(s, "\t", "")
	s = strings.TrimLeft(s, "\n")
	return []byte(s)
}

func TestObjectFileNameRegexp(t *testing.T) {
	assert.Equal(t, ObjectFileNameRegexp.MatchString("i.runtime.front.yaml"), true)
	assert.Equal(t, ObjectFileNameRegexp.MatchString("i.runtime.super-front.yaml"), true)
	assert.Equal(t, ObjectFileNameRegexp.MatchString("i.runtime.super-front.patch.yaml"), true)
	assert.Equal(t, ObjectFileNameRegexp.MatchString("i.runtime.super-front.notpatch.yaml"), false)
	assert.Equal(t, ObjectFileNameRegexp.MatchString("s.runtime.super-front.patch.yaml"), false)

	matches := ObjectFileNameRegexp.FindStringSubmatch("i.runtime.front.yaml")
	assert.Equal(t, len(matches), 4)
	assert.Equal(t, matches[1], "runtime")
	assert.Equal(t, matches[2], "front")
	assert.Equal(t, matches[3], "")

	matches = ObjectFileNameRegexp.FindStringSubmatch("i.runtime.front.patch.yaml")
	assert.Equal(t, len(matches), 4)
	assert.Equal(t, matches[1], "runtime")
	assert.Equal(t, matches[2], "front")
	assert.Equal(t, matches[3], "patch")
}

func TestFillKubeObjectRequiredFields(t *testing.T) {
	protoparser := &protojson.UnmarshalOptions{DiscardUnknown: true}
	ObjKind := "Kind"
	ObjName := "name"
	t.Run("case-w-empty-data", func(t *testing.T) {
		content := makeValidYamlContent(`
		spec:
		  test: 1`)
		data := &manifest.ResourceManifest{}
		jsonContent, err := yaml.YAMLToJSON(content)
		assert.NoError(t, err)

		err = protoparser.Unmarshal(jsonContent, data)
		assert.NoError(t, err)

		err = fillKubeObjectRequiredFields(data, APIVersion, ObjKind, ObjName)
		assert.NoError(t, err)
		assert.Equal(t, data.ApiVersion, APIVersion)
		assert.Equal(t, data.Kind, ObjKind)
		assert.Equal(t, data.Metadata.Fields["name"].GetStringValue(), ObjName)
	})
	t.Run("case-w-metadata", func(t *testing.T) {
		content := makeValidYamlContent(`
		spec:
		  test: 1
		apiVersion: otherApiVersion
		kind: notKind
		metadata:
		  namespace: namespace
		`)
		data := &manifest.ResourceManifest{}
		jsonContent, err := yaml.YAMLToJSON(content)
		assert.NoError(t, err)

		err = protoparser.Unmarshal(jsonContent, data)
		assert.NoError(t, err)
		err = fillKubeObjectRequiredFields(data, APIVersion, ObjKind, ObjName)
		assert.NoError(t, err)
		assert.Equal(t, data.ApiVersion, "otherApiVersion")
		assert.Equal(t, data.Kind, "notKind")
		assert.Equal(t, data.Metadata.Fields["name"].GetStringValue(), ObjName)
		assert.Equal(t, data.Metadata.Fields["namespace"].GetStringValue(), "namespace")
	})
}

func TestNewResourceFile(t *testing.T) {
	content := makeValidYamlContent(`
	metadata:
	  namespace: test
	spec:
	  test: 1
	  data:
	    message: hallow
	patches:
	- kustomize:
	other_data: 1`)
	kind := "Runtime"
	name := "test"
	f, err := NewResourceFile("/test", "any.yaml", kind, name, []byte(content))
	assert.NoError(t, err)
	assert.Equal(t, f.Data.Metadata.Fields["name"].GetStringValue(), name)
	assert.Equal(t, f.Data.ApiVersion, APIVersion)
	assert.Equal(t, f.Data.Kind, kind)
}

func TestResourceFile_GenerateResourceBaseInline(t *testing.T) {
	content := makeValidYamlContent(`
    metadata:
      namespace: test
    spec:
      test: 1
      data:
        message: hallow
    patches:
    - kustomize:
    other_data: 1`)
	kind := "Runtime"
	name := "test"
	f, err := NewResourceFile("/test", "any.yaml", kind, name, []byte(content))
	assert.NoError(t, err)

	res, err := f.GenerateResourceBaseInline()
	assert.NoError(t, err)
	expected := makeValidYamlContent(`
	apiVersion: k.yandex-team.ru/v1
	kind: Runtime
	metadata:
	  name: test
	  namespace: test
	spec:
	  data:
	    message: hallow
	  test: 1
	`)
	assert.Equal(t, res, string(expected))
}

func TestResourceFile_GenerateKustomizePatch(t *testing.T) {
	t.Run("case-w-spec", func(t *testing.T) {
		content := makeValidYamlContent(`
		metadata:
		  namespace: test
		spec:
		  test: 1
		  data:
		    message: hallow
		patches:
		- kustomize:
		other_data: 1`)
		kind := "Runtime"
		name := "test"
		namePrefix := "prefix-"
		f, err := NewResourceFile("/test", "any.yaml", kind, name, []byte(content))
		assert.NoError(t, err)

		patch, err := f.GenerateKustomizePatch(namePrefix)
		assert.NoError(t, err)
		assert.Equal(t, patch.Kustomize.Fields[namePrefixFieldName].GetStringValue(), namePrefix)
		merges := patch.Kustomize.Fields[patchesStrategicMergeFiledName].GetListValue().AsSlice()
		assert.Equal(t, len(merges), 1)
		expectedMergeContent, err := f.GenerateResourceBaseInline()
		assert.NoError(t, err)
		assert.Equal(t, merges[0].(string), expectedMergeContent)
	})

	t.Run("case-wo-spec", func(t *testing.T) {
		content := makeValidYamlContent(`
		kind: Runtime`)
		kind := "Runtime"
		name := "test"
		namePrefix := "prefix-"
		f, err := NewResourceFile("/test", "any.yaml", kind, name, []byte(content))
		assert.NoError(t, err)

		patch, err := f.GenerateKustomizePatch(namePrefix)
		assert.NoError(t, err)
		assert.Equal(t, patch.Kustomize.Fields[namePrefixFieldName].GetStringValue(), namePrefix)
		merges := patch.Kustomize.Fields[patchesStrategicMergeFiledName].GetListValue().AsSlice()
		assert.Equal(t, len(merges), 0)
	})
}

func TestResourceFile_GeneratePatches(t *testing.T) {
	content := makeValidYamlContent(`
	spec:
	  test: 1
	patches:
	- envSource: env.properties
	- files:
	    /configs/cfg.yml: cfg.yml
	- kustomize:
	    namePrefix: this-is-other-kustomise`)
	kind := "Runtime"
	name := "test"
	namePrefix := "prefix-"
	f, err := NewResourceFile("/test", "any.yaml", kind, name, []byte(content))
	assert.NoError(t, err)

	patches, err := f.GeneratePatches(namePrefix, true)
	assert.NoError(t, err)
	assert.Equal(t, len(patches), 4)
	assert.Equal(t, patches[0].Kustomize.Fields[namePrefixFieldName].GetStringValue(), namePrefix)
	assert.Equal(t, patches[3].Kustomize.Fields[namePrefixFieldName].GetStringValue(), "this-is-other-kustomise")
}
