package iyaml

import (
	"fmt"
	"path"
	"strings"

	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/cli/commands/make/manifest"
	"a.yandex-team.ru/infra/infractl/cli/internal/serialization"
	"a.yandex-team.ru/infra/infractl/util/kubeutil"
)

type buildResult struct {
	objects         []client.Object
	objectKeys      map[string]bool
	output          string
	foundNamespaces map[string]bool
}

func newBuildResult() *buildResult {
	return &buildResult{
		objectKeys:      map[string]bool{},
		foundNamespaces: map[string]bool{},
	}
}

func makeBuildResultFromManifest(manifestContent *manifest.Manifest) *buildResult {
	br := newBuildResult()
	ns := manifestContent.GetNamespace()
	if ns != "" {
		br.foundNamespaces[ns] = true
	}
	return br
}

func makeBuildResultFromObjects(objects []client.Object, foundNamespaces map[string]bool) (*buildResult, error) {
	r := newBuildResult()
	r.appendNamespaces(foundNamespaces)
	if err := r.appendObjects(objects...); err != nil {
		return nil, err
	}
	return r, nil
}

func (br *buildResult) GetObjects() []client.Object {
	return br.objects
}

func (br *buildResult) GetFoundNamespaces() []string {
	keys := make([]string, 0, len(br.foundNamespaces))
	for k := range br.foundNamespaces {
		keys = append(keys, k)
	}
	return keys
}

func (br *buildResult) appendNamespaces(foundNamespaces map[string]bool) {
	for key := range foundNamespaces {
		br.foundNamespaces[key] = true
	}
}

func (br *buildResult) validateObjectKey(key string) error {
	if _, ok := br.objectKeys[key]; ok {
		return fmt.Errorf("duplicate objects %s", key)
	}
	return nil
}

func (br *buildResult) appendObjects(objects ...client.Object) error {
	for _, object := range objects {
		objectKey := fmt.Sprintf(
			"%s/%s",
			object.GetObjectKind().GroupVersionKind().Kind,
			client.ObjectKeyFromObject(object),
		)
		if err := br.validateObjectKey(objectKey); err != nil {
			return err
		}
		br.objectKeys[objectKey] = true
		br.objects = append(br.objects, object)
	}

	return nil
}

func (br *buildResult) appendBuildResult(newBR *buildResult) error {
	for objectKey := range newBR.objectKeys {
		if err := br.validateObjectKey(objectKey); err != nil {
			return err
		}
		br.objectKeys[objectKey] = true
	}

	br.objects = append(br.objects, newBR.objects...)
	br.appendNamespaces(newBR.foundNamespaces)
	return nil
}

type Patcher interface {
	Patch(objects []client.Object) ([]client.Object, error)
}

type manifestProcessor struct {
	arcRoot    string
	KubeClient kubeutil.Client
}

func NewManifestProcessor(arcRoot string, kubeClient kubeutil.Client) *manifestProcessor {
	return &manifestProcessor{
		arcRoot:    arcRoot,
		KubeClient: kubeClient,
	}
}

func (p *manifestProcessor) processResource(dir string, result *buildResult, r *manifest.Resource) error {
	switch r.Path.(type) {
	case *manifest.Resource_Base:
		br, err := p.Build(path.Join(p.arcRoot, r.GetBase()))
		if err != nil {
			return err
		}
		return result.appendBuildResult(br)

	case *manifest.Resource_File:
		objects, err := p.parseYamlResource(path.Join(dir, r.GetFile()))
		if err != nil {
			return err
		}
		return result.appendObjects(objects...)
	case *manifest.Resource_Inline:
		objects, err := p.parseYamlContentResource(r.GetInline())
		if err != nil {
			return err
		}
		return result.appendObjects(objects...)
	}
	return fmt.Errorf("unknown resource type")
}

func (p *manifestProcessor) inferResourcesAndPatches(client kubeutil.Client, dir, arcRoot string, manifestContent *manifest.Manifest) error {
	var err error
	resourceFiles, err := collectResources(client, dir)
	if err != nil {
		return err
	}

	arcadiaCIFileExisits, err := isFileExists(path.Join(dir, ArcadiaCIFileName))
	if err != nil {
		return err
	}

	if IsDiscoveryBaseNeeded(manifestContent, arcadiaCIFileExisits) {
		appendBase(dir, arcRoot, manifestContent)
	}

	if IsDiscoveryFilesNeeded(manifestContent) {
		err = p.autoFillManifestResources(resourceFiles, manifestContent)
		if err != nil {
			return fmt.Errorf("autoFillManifestResources: %w", err)
		}
	}

	if IsDiscoveryPatchesNeeded(manifestContent) {
		err = p.autoFillManifestPatches(dir, resourceFiles, manifestContent)
		if err != nil {
			return fmt.Errorf("autoFillManifestPatches: %w", err)
		}
	}

	return nil
}

func (p *manifestProcessor) Build(dir string) (*buildResult, error) {
	manifestContent, err := p.makeManifest(p.KubeClient, dir)
	if err != nil {
		return nil, err
	}

	result := makeBuildResultFromManifest(manifestContent)

	for _, r := range manifestContent.Resources {
		if err = p.processResource(dir, result, r); err != nil {
			return nil, fmt.Errorf("cannot process resource %q: %w", r, err)
		}
	}

	for i, patch := range manifestContent.Patches {
		if result, err = p.processPatch(patch, result, dir); err != nil {
			return nil, fmt.Errorf("cannot apply patch #%d: %w", i, err)
		}
	}

	return result, nil
}

func (p *manifestProcessor) processPatch(
	patch *manifest.Patch,
	input *buildResult,
	dir string,
) (*buildResult, error) {
	patchers := []Patcher{}

	if len(patch.Files) != 0 || len(patch.Dirs) != 0 {
		patchers = append(patchers, NewInsertFilesPatcher(
			patch.Files, patch.Dirs, patch.Selector, dir,
		))
	}
	if len(patch.Env) != 0 || len(patch.EnvSource) != 0 {
		patchers = append(patchers, NewEnvPatcher(
			patch.Env, patch.EnvSource, patch.Selector, dir,
		))
	}
	if patch.Kustomize != nil {
		if len(patchers) > 0 {
			return nil, fmt.Errorf(
				"if kustomize is in patch, all other fileds must be empty",
			)
		}
		patchers = append(patchers, NewKustomizePatcher(
			patch.Kustomize, dir, p.arcRoot,
		))
	}

	if len(patchers) == 0 {
		return nil, fmt.Errorf("all patch fields are empty")
	}

	var err error
	objects := input.objects
	for _, patcher := range patchers {
		objects, err = patcher.Patch(objects)
		if err != nil {
			return nil, fmt.Errorf("patch processing failed: %w", err)
		}
	}

	return makeBuildResultFromObjects(objects, input.foundNamespaces)
}

func (p *manifestProcessor) parseYamlResource(filePath string) ([]client.Object, error) {
	objects, err := serialization.ParseYamlTyped(filePath)
	if err != nil {
		return nil, fmt.Errorf("unable to parse resource file %s: %w", filePath, err)
	}
	return objects, nil
}

func (p *manifestProcessor) parseYamlContentResource(content string) ([]client.Object, error) {
	objects, err := serialization.ParseReaderTyped(strings.NewReader(content))
	if err != nil {
		return nil, fmt.Errorf("unable to parse resource content: %w", err)
	}
	return objects, nil
}

func (p *manifestProcessor) SetNamespaceIfNeeded(br *buildResult) error {
	namespaces := br.GetFoundNamespaces()
	if len(namespaces) == 0 {
		// No namespaces set in i.yaml files, so let's just presume
		// that all objects already have namespace in their own spec,
		// or they do not need particular namespace given in spec at all
		return nil
	}
	for _, o := range br.GetObjects() {
		if o.GetNamespace() != "" {
			continue
		}
		if len(namespaces) > 1 {
			kind := o.GetObjectKind().GroupVersionKind().Kind
			name := o.GetName()
			return fmt.Errorf("cannot infer namespace for %v %q, %v different namespaces found in base i.yaml files: %q", kind, name, len(namespaces), namespaces)
		}
		o.SetNamespace(namespaces[0])
	}
	return nil
}
