package serialization

import (
	"encoding/json"
	"fmt"
	"io"
	"log"
	"os"

	corev1 "k8s.io/api/core/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/runtime/schema"
	"k8s.io/apimachinery/pkg/util/yaml"
	"k8s.io/client-go/kubernetes/scheme"
	"sigs.k8s.io/controller-runtime/pkg/client"

	abv1 "a.yandex-team.ru/infra/infractl/controllers/awacs/api/backend/v1"
	auv1 "a.yandex-team.ru/infra/infractl/controllers/awacs/api/upstream/v1"
	dv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	rv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
)

const bufferSize = 65536

// Unknown represents an unparsed Kubernetes resource with an interpreted
// TypeMeta and ObjectMeta fields which are used for type recognition.
// +k8s:deepcopy-gen:interfaces=k8s.io/apimachinery/pkg/runtime.Object
type Unknown struct {
	unknown
	Raw json.RawMessage `json:",inline"`
}

type unknown struct {
	runtime.TypeMeta
	Metadata metav1.ObjectMeta `json:"metadata"`
}

type EmptyObject struct{}

func (e *EmptyObject) Error() string {
	return "Object is empty"
}

func IsEmptyObjectError(err error) bool {
	_, ok := err.(*EmptyObject)
	return ok
}

// GetObjectKind returns the ObjectKind for this Unknown.
// Implements runtime.Object
func (r *Unknown) GetObjectKind() schema.ObjectKind {
	return &r.TypeMeta
}

// UnmarshalJSON consumes the specified data as a binary blob w/o interpreting it
func (r *Unknown) UnmarshalJSON(data []byte) error {
	if err := json.Unmarshal(data, &r.unknown); err != nil {
		return err
	}
	if err := r.Raw.UnmarshalJSON(data); err != nil {
		return err
	}
	return nil
}

// MarshalJSON returns the raw message
func (r *Unknown) MarshalJSON() ([]byte, error) {
	return r.Raw.MarshalJSON()
}

type UniversalDecoder struct {
	runtime.Decoder
	streamDecoder *yaml.YAMLOrJSONDecoder
}

func newUniversalDecoder(r io.Reader) *UniversalDecoder {
	bufReader, _, _ := yaml.GuessJSONStream(r, bufferSize)
	streamDecoder := yaml.NewYAMLOrJSONDecoder(bufReader, bufferSize)
	decoder := scheme.Codecs.UniversalDeserializer()
	universalDecoder := &UniversalDecoder{
		Decoder:       decoder,
		streamDecoder: streamDecoder,
	}

	return universalDecoder
}
func (d *UniversalDecoder) Decode() (object runtime.Object, err error) {
	var unk Unknown
	if err = d.streamDecoder.Decode(&unk); err != nil {
		return
	}
	if len(unk.Raw) == 0 {
		return nil, &EmptyObject{}
	}
	if unk.Kind == "" {
		return nil, fmt.Errorf("unparsable object found")
	}
	object, err = runtime.Decode(d.Decoder, unk.Raw)
	if err != nil {
		return nil, err
	}
	return
}

func Decode(r io.Reader) (objects []runtime.Object, err error) {
	decoder := newUniversalDecoder(r)

	for {
		object, err := decoder.Decode()
		if err != nil {
			if err == io.EOF {
				break
			}
			if IsEmptyObjectError(err) {
				continue
			}
			return nil, fmt.Errorf("decode failure: %v", err)
		}
		objects = append(objects, object)
	}
	return objects, nil
}

func brokenKind(kind string) {
	log.Fatalf("Failed to parse %v spec properly, infractl is probably broken, report to https://st.yandex-team.ru/INFRACTL", kind)
}

func ParseObjectKind(object runtime.Object) (client.Object, error) {
	kind := object.GetObjectKind()

	// Not sure if it is necessary, or that we should not compare all the GKV,
	// but for now let's do quick sanity check
	switch kind.GroupVersionKind().Kind {
	case "Namespace":
		if parsedObject, ok := object.(*corev1.Namespace); ok {
			return parsedObject, nil
		}
		brokenKind("Namespace")
	case "Runtime":
		if parsedObject, ok := object.(*rv1.Runtime); ok {
			return parsedObject, nil
		}
		brokenKind("Runtime")
	case "DeployStage":
		if parsedObject, ok := object.(*dv1.DeployStage); ok {
			return parsedObject, nil
		}
		brokenKind("DeployStage")
	case "AwacsUpstream":
		if parsedObject, ok := object.(*auv1.AwacsUpstream); ok {
			return parsedObject, nil
		}
		brokenKind("AwacsUpstream")
	case "AwacsBackend":
		if parsedObject, ok := object.(*abv1.AwacsBackend); ok {
			return parsedObject, nil
		}
		brokenKind("AwacsBackend")
	default:
		return nil, fmt.Errorf("k8s Kind %q is not supported", kind.GroupVersionKind())
	}

	// compiler asks for return after log.Fatalf for some reason
	return nil, nil
}

func ParseReader(r io.Reader) (objects []runtime.Object, err error) {
	objects, err = Decode(r)
	if err != nil {
		objects = nil
		return
	}

	for _, object := range objects {
		if _, err = ParseObjectKind(object); err != nil {
			objects = nil
			return
		}
	}

	return
}

func ParseReaderTyped(r io.Reader) ([]client.Object, error) {
	objects, err := ParseReader(r)
	if err != nil {
		return nil, err
	}
	parsedObjects := make([]client.Object, 0, len(objects))
	for _, object := range objects {
		parsedObject, err := ParseObjectKind(object)
		if err != nil {
			return nil, err
		}
		parsedObjects = append(parsedObjects, parsedObject)
	}

	return parsedObjects, nil
}

func ParseYaml(path string) ([]runtime.Object, error) {
	f, err := os.Open(path)
	if err != nil {
		return nil, err
	}
	defer f.Close()

	objects, err := ParseReader(f)
	if err != nil {
		return nil, fmt.Errorf("in file %q: %w", path, err)
	}
	return objects, nil
}

func ParseYamlTyped(path string) ([]client.Object, error) {
	f, err := os.Open(path)
	if err != nil {
		return nil, err
	}
	defer f.Close()

	return ParseReaderTyped(f)
}

func MustReadOneObject(path string) runtime.Object {
	objects, err := ParseYaml(path)
	if err != nil {
		log.Fatalf("failed to parse file %q: %v", path, err)
	}
	if len(objects) != 1 {
		log.Fatalf("file %q must contain exactly one object, %v found", path, len(objects))
	}

	return objects[0]
}
