package serialization

import (
	"bytes"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"os"

	"github.com/ghodss/yaml"
	"k8s.io/apimachinery/pkg/runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"
)

const YamlDocumentSeparator = "---\n"

type YamlEncoder struct {
	w              io.Writer
	encodedObjects int
}

func NewYamlEncoder(w io.Writer) *YamlEncoder {
	return &YamlEncoder{w: w}
}

func (e *YamlEncoder) Encode(obj runtime.Object) error {
	dump, err := yaml.Marshal(obj)
	if err != nil {
		return err
	}

	if e.encodedObjects > 0 {
		if _, err := e.w.Write([]byte(YamlDocumentSeparator)); err != nil {
			return err
		}
	}

	if _, err := e.w.Write(dump); err != nil {
		return err
	}

	e.encodedObjects += 1
	return nil
}

func DumpToFile(object runtime.Object, path string) error {
	dump, err := yaml.Marshal(object)
	if err != nil {
		return err
	}
	err = ioutil.WriteFile(path, dump, 0o600)
	return err
}

func MustDumpObject(obj client.Object, outputFile string) {
	dump, err := yaml.Marshal(obj)
	if err != nil {
		log.Fatalf("Object dump failed: %v", err)
	}

	if outputFile == "-" {
		fmt.Println(string(dump) + "\n")
	} else {
		err = ioutil.WriteFile(outputFile, dump, 0o644)
		if err != nil {
			fmt.Fprintf(os.Stderr, "Failed to open %q for writing: %v\nWill dump spec to stderr:\n\n%v\n\n", outputFile, err, string(dump))
		}
	}
}

func DumpObjectsToFile[T runtime.Object](objects []T, path string) error {
	buf := bytes.Buffer{}
	enc := NewYamlEncoder(&buf)
	for _, object := range objects {
		if err := enc.Encode(object); err != nil {
			return err
		}
	}

	return ioutil.WriteFile(path, buf.Bytes(), 0o755)
}

func DumpObjectsToStdout[T runtime.Object](objects []T) error {
	enc := NewYamlEncoder(os.Stdout)
	for _, object := range objects {
		if err := enc.Encode(object); err != nil {
			return err
		}
	}
	return nil
}
