package substitutio

import (
	"fmt"

	"gopkg.in/yaml.v3"
	"k8s.io/apimachinery/pkg/runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/cli/internal/deploystage"
	dv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	rv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	"a.yandex-team.ru/yp/go/proto/podagent"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

func AugmentSpec(unspec runtime.Object, substs *Substitutions) error {
	templateDump, err := DumpTemplate(unspec)
	if err != nil {
		return err
	}
	substsDump, err := yaml.Marshal(substs)
	if err != nil {
		return err
	}

	cspec := unspec.(client.Object)
	if cspec == nil {
		return nil
	}
	annotations := cspec.GetAnnotations()
	if annotations == nil {
		annotations = map[string]string{}
	}
	annotations[TemplateAnnotation] = string(templateDump[:])
	annotations[SubstitutionsAnnotation] = string(substsDump[:])
	cspec.SetAnnotations(annotations)

	switch spec := unspec.(type) {
	case *rv1.Runtime:
		// TODO (torkve) substitute volumes
		// TODO (torkve) substitute static resources

		path := TryParseDockerImage(spec.Spec.Image)
		if path != nil {
			image, err := substs.GetDockerImage(*path)
			if err != nil {
				return err
			}
			spec.Spec.Image = image
			break
		}

		for _, layer := range spec.Spec.Layers {
			resourceRef := TryParseSandboxResource(layer.ResourceId)
			if resourceRef != nil {
				layerInfo, err := substs.GetSandboxResource(*resourceRef)
				if err != nil {
					return err
				}
				layer.ResourceId = layerInfo.ResourceID
			}
		}
	case *dv1.DeployStage:
		buildManifestString := annotations[BuildManifestAnnotation]
		if len(buildManifestString) == 0 {
			return nil
		}
		manifest, err := LoadBuildManifest(buildManifestString)
		if err != nil {
			return err
		}

		for _, dockerPackage := range manifest.DockerPackages {
			_, du, err := deploystage.DeduceDeployUnit(spec.Spec.StageSpec, dockerPackage.DeployUnitID)
			if err != nil {
				return err
			}

			boxID, _, err := deploystage.DeduceBox(du, dockerPackage.BoxID)
			if err != nil {
				return err
			}

			image, err := rv1.SplitDockerImage(substs.GetDockerImage(dockerPackage.Path))
			if err != nil {
				return err
			}
			if du.ImagesForBoxes == nil {
				du.ImagesForBoxes = map[string]*ypapi.TDockerImageDescription{}
			}
			du.ImagesForBoxes[boxID] = image
		}

		for _, layer := range manifest.Layers {
			_, du, err := deploystage.DeduceDeployUnit(spec.Spec.StageSpec, layer.DeployUnitID)
			if err != nil {
				return err
			}
			layerInfo, err := substs.GetSandboxResource(layer.Path)
			if err != nil {
				return err
			}

			err = insertLayer(du, layer.LayerID, layerInfo)
			if err != nil {
				return err
			}
		}

		for _, staticResource := range manifest.StaticResources {
			_, du, err := deploystage.DeduceDeployUnit(spec.Spec.StageSpec, staticResource.DeployUnitID)
			if err != nil {
				return err
			}
			staticResourceInfo, err := substs.GetSandboxResource(staticResource.Path)
			if err != nil {
				return err
			}

			err = insertStaticResource(du, staticResource.StaticResourceID, staticResourceInfo)
			if err != nil {
				return err
			}
		}
	default:
		return nil
	}
	return nil
}

func insertLayer(
	du *ypapi.TDeployUnitSpec,
	layerID string,
	layerInfo *SandboxResource,
) error {
	var pts *ypapi.TPodTemplateSpec
	switch primitive := du.PodDeployPrimitive.(type) {
	case *ypapi.TDeployUnitSpec_ReplicaSet:
		pts = primitive.ReplicaSet.GetReplicaSetTemplate().GetPodTemplateSpec()
	case *ypapi.TDeployUnitSpec_MultiClusterReplicaSet:
		pts = primitive.MultiClusterReplicaSet.GetReplicaSet().GetPodTemplateSpec()
	}
	spec := pts.GetSpec().GetPodAgentPayload().GetSpec()
	if spec == nil {
		return fmt.Errorf("deploy unit is malformed")
	}

	for _, l := range spec.Resources.Layers {
		if l.Id != layerID {
			continue
		}
		if l.Meta != nil || l.DownloadMethod != nil {
			return fmt.Errorf("layer %q already contains resource info", layerID)
		}

		attributes := make(map[string]string, len(layerInfo.Attributes))
		for k, v := range layerInfo.Attributes {
			attributes[k] = fmt.Sprint(v)
		}

		l.Meta = &podagent.TResourceMeta{Meta: &podagent.TResourceMeta_SandboxResource{
			SandboxResource: &podagent.TSandboxResource{
				TaskType:     layerInfo.TaskType,
				TaskId:       layerInfo.TaskID,
				ResourceType: layerInfo.ResourceType,
				ResourceId:   layerInfo.ResourceID,
				Attributes:   attributes,
			},
		}}
		l.Checksum = "EMPTY:"
		l.DownloadMethod = &podagent.TLayer_Url{Url: layerInfo.MakeResourceURL()}
	}

	return nil
}

func insertStaticResource(
	du *ypapi.TDeployUnitSpec,
	staticResourceID string,
	staticResourceInfo *SandboxResource,
) error {
	var pts *ypapi.TPodTemplateSpec
	switch primitive := du.PodDeployPrimitive.(type) {
	case *ypapi.TDeployUnitSpec_ReplicaSet:
		pts = primitive.ReplicaSet.GetReplicaSetTemplate().GetPodTemplateSpec()
	case *ypapi.TDeployUnitSpec_MultiClusterReplicaSet:
		pts = primitive.MultiClusterReplicaSet.GetReplicaSet().GetPodTemplateSpec()
	}
	spec := pts.GetSpec().GetPodAgentPayload().GetSpec()
	if spec == nil {
		return fmt.Errorf("deploy unit is malformed")
	}

	for _, staticResource := range spec.Resources.StaticResources {
		if staticResource.Id != staticResourceID {
			continue
		}
		if staticResource.Meta != nil || staticResource.DownloadMethod != nil {
			return fmt.Errorf("static resource %q already contains resource info", staticResourceID)
		}

		attributes := make(map[string]string, len(staticResourceInfo.Attributes))
		for k, v := range staticResourceInfo.Attributes {
			attributes[k] = fmt.Sprint(v)
		}

		staticResource.Meta = &podagent.TResourceMeta{Meta: &podagent.TResourceMeta_SandboxResource{
			SandboxResource: &podagent.TSandboxResource{
				TaskType:     staticResourceInfo.TaskType,
				TaskId:       staticResourceInfo.TaskID,
				ResourceType: staticResourceInfo.ResourceType,
				ResourceId:   staticResourceInfo.ResourceID,
				Attributes:   attributes,
			},
		}}
		staticResource.DownloadMethod = &podagent.TResource_Url{Url: staticResourceInfo.MakeResourceURL()}
	}

	return nil
}
