package substitutio

import (
	"fmt"
	"os"
	"regexp"
	"strings"

	"gopkg.in/yaml.v3"
)

var (
	packageRe  = regexp.MustCompile(`^\$\{package:([a-z0-9_\.-]+(?:/[a-z0-9_\.-]+)*)\}$`)
	resourceRe = regexp.MustCompile(`^\$\{resource:([a-z0-9_\.-]+(?:/[a-z0-9_\.-]+)*)\}$`)
)

type SandboxResource struct {
	TaskType     string                 `yaml:"task_type" json:"task_type"`
	TaskID       string                 `yaml:"task_id" json:"task_id"`
	ResourceType string                 `yaml:"resource_type" json:"resource_type"`
	ResourceID   string                 `yaml:"resource_id" json:"resource_id"`
	Attributes   map[string]interface{} `yaml:"attributes" json:"attributes"`
}

func (r *SandboxResource) MakeResourceURL() string {
	return fmt.Sprintf("sbr:%s", r.ResourceID)
}

type Substitutions struct {
	DockerPackages   map[string]string           `yaml:"docker_packages,omitempty" json:"docker_packages,omitempty"`
	SandboxResources map[string]*SandboxResource `yaml:"sandbox_resources,omitempty" json:"sandbox_resources,omitempty"`
}

func NewSubstitutions() *Substitutions {
	return &Substitutions{
		DockerPackages:   map[string]string{},
		SandboxResources: map[string]*SandboxResource{},
	}
}

func (s *Substitutions) Merge(other *Substitutions, strict bool) error {
	for id, dockerPackage := range other.DockerPackages {
		old, found := s.DockerPackages[id]
		if found && strict && old != dockerPackage {
			return fmt.Errorf("substitutions contain differing docker package for %q", id)
		} else {
			s.DockerPackages[id] = dockerPackage
		}
	}
	for id, resource := range other.SandboxResources {
		old, found := s.SandboxResources[id]
		if found && strict && old.ResourceID != resource.ResourceID {
			return fmt.Errorf("substitutions contain differing sandbox resource for %q", id)
		} else {
			s.SandboxResources[id] = resource
		}
	}
	return nil
}

func TryParseDockerImage(field string) *string {
	match := packageRe.FindStringSubmatch(field)
	if match != nil {
		return &match[1]
	}
	return nil
}

func TryParseSandboxResource(field string) *string {
	match := resourceRe.FindStringSubmatch(field)
	if match != nil {
		return &match[1]
	}
	return nil
}

func (s *Substitutions) GetDockerImage(path string) (image string, err error) {
	val, ok := s.DockerPackages[path]
	if !ok {
		return "", fmt.Errorf("artifact not found for docker image %q", path)
	}
	return val, nil
}

func (s *Substitutions) GetSandboxResource(path string) (resource *SandboxResource, err error) {
	val, ok := s.SandboxResources[path]
	if !ok {
		return nil, fmt.Errorf("artifact not found sandbox resource %q", path)
	}
	return val, nil
}

func (s *Substitutions) DumpToFile(path string) error {
	f, err := os.Create(path)
	defer func(f *os.File) {
		_ = f.Close()
	}(f)
	if err != nil {
		return err
	}

	encoder := yaml.NewEncoder(f)
	encoder.SetIndent(2)
	err = encoder.Encode(s)
	if err != nil {
		return err
	}
	err = encoder.Close()
	return err
}

func (s *Substitutions) DumpToString() (value string, err error) {
	var out strings.Builder
	encoder := yaml.NewEncoder(&out)
	encoder.SetIndent(2)
	err = encoder.Encode(s)
	if err != nil {
		return
	}
	err = encoder.Close()
	if err != nil {
		return
	}
	value = out.String()
	return
}

func LoadSubstitutions(annotation string) (substitutions *Substitutions, err error) {
	substitutions = &Substitutions{}
	err = yaml.Unmarshal([]byte(annotation), substitutions)
	if err != nil {
		substitutions = nil
	}
	if substitutions.DockerPackages == nil {
		substitutions.DockerPackages = map[string]string{}
	}
	if substitutions.SandboxResources == nil {
		substitutions.SandboxResources = map[string]*SandboxResource{}
	}
	return
}

func Load(path string) *Substitutions {
	ret := NewSubstitutions()

	f, err := os.Open(path)
	defer func(f *os.File) {
		_ = f.Close()
	}(f)

	if err == nil {
		decoder := yaml.NewDecoder(f)
		_ = decoder.Decode(ret)
	}
	if ret.DockerPackages == nil {
		ret.DockerPackages = map[string]string{}
	}
	if ret.SandboxResources == nil {
		ret.SandboxResources = map[string]*SandboxResource{}
	}

	return ret
}
